<?php
/**
 * RedParts sputnik search.
 *
 * @package RedParts\Sputnik
 * @since 1.1.0
 * @noinspection SqlResolve
 */

namespace RedParts\Sputnik;

use RedParts\Sputnik\WPML\WPML;
use WP_Query;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Search' ) ) {
	/**
	 * Class Search.
	 *
	 * @package RedParts\Sputnik
	 */
	class Search extends Singleton {
		/**
		 * Initialization.
		 */
		public function init() {
			add_action( 'after_setup_theme', array( $this, 'deferred_init' ) );
		}

		/**
		 * Deferred initialization.
		 */
		public function deferred_init() {
			if ( ! class_exists( 'WooCommerce' ) ) {
				return;
			}

			add_action( 'wp_ajax_redparts_sputnik_search_suggestions', array( $this, 'ajax_search_suggestions' ) );
			add_action( 'wp_ajax_nopriv_redparts_sputnik_search_suggestions', array( $this, 'ajax_search_suggestions' ) );

			add_filter( 'posts_search', array( $this, 'wp_query_posts_search' ), 10, 2 );
			add_filter( 'posts_join', array( $this, 'wp_query_posts_join' ), 10, 2 );
		}

		/**
		 * Modifies search part of the WP_Query.
		 *
		 * @since 1.12.0
		 *
		 * @noinspection PhpMissingParamTypeInspection,PhpMissingReturnTypeInspection
		 *
		 * @param string   $search - Search part of the SQL query.
		 * @param WP_Query $query  - WordPress query object.
		 *
		 * @return string
		 */
		public function wp_query_posts_search( $search, $query ) {
			global $wpdb;

			$is_search_by_products = $query->is_search
				&& 'product' === $query->get( 'post_type' )
				&& ! empty( $query->query_vars['search_terms'] );

			if ( ! $is_search_by_products ) {
				return $search;
			}

			$terms = $query->query_vars['search_terms'];

			if ( 'yes' === Settings::instance()->get( 'search_by_sku' ) ) {
				$offset = 0;
				$i      = 0;

				while ( true ) {
					$example = "($wpdb->posts.post_title LIKE '";

					$pos = strpos( $search, $example, $offset );

					if ( false === $pos || ! isset( $terms[ $i ] ) ) {
						break;
					}

					$match_type = Settings::instance()->get( 'search_by_sku_match' );

					if ( 'starts' === $match_type ) {
						$value = $wpdb->esc_like( $terms[ $i ] ) . '%';
					} elseif ( 'contains' === $match_type ) {
						$value = '%' . $wpdb->esc_like( $terms[ $i ] ) . '%';
					} else {
						$value = $wpdb->esc_like( $terms[ $i ] );
					}

					$new = $wpdb->prepare( "(redparts_sputnik_pm1.meta_value LIKE %s) OR ($wpdb->posts.post_title LIKE '", $value );

					$search = substr_replace( $search, $new, $pos, strlen( $example ) );

					$offset += strlen( $new );
					++$i;
				}
			}

			$exact_match = (array) Settings::instance()->get( 'search_by_attributes' );
			$exact_match = array_map( 'wc_attribute_taxonomy_name', $exact_match );

			$starts_with = (array) Settings::instance()->get( 'search_by_attributes_starts' );
			$starts_with = array_map( 'wc_attribute_taxonomy_name', $starts_with );

			$contains = (array) Settings::instance()->get( 'search_by_attributes_contains' );
			$contains = array_map( 'wc_attribute_taxonomy_name', $contains );

			$attributes = array_merge(
				$exact_match,
				$starts_with,
				$contains
			);

			if ( ! empty( $attributes ) ) {
				$prepare_where = function( $attributes, $match_type ) use ( $wpdb, $terms ) {
					$sql_terms = array();

					foreach ( $terms as $term ) {
						if ( 'starts' === $match_type ) {
							$value = $wpdb->esc_like( $term ) . '%';
						} elseif ( 'contains' === $match_type ) {
							$value = '%' . $wpdb->esc_like( $term ) . '%';
						} else {
							$value = $wpdb->esc_like( $term );
						}

						$sql_terms[] = $wpdb->prepare( 't.name LIKE %s', $value );
					}

					$sql_terms  = implode( ' OR ', $sql_terms );
					$taxonomies = $attributes;

					$taxonomies = array_map(
						function( $taxonomy ) use ( $wpdb ) {
							return $wpdb->prepare( '%s', $taxonomy );
						},
						$taxonomies
					);
					$taxonomies = implode( ', ', $taxonomies );

					return "(($sql_terms) AND tt.taxonomy IN ($taxonomies))";
				};

				$where = array();

				if ( ! empty( $exact_match ) ) {
					$where[] = $prepare_where( $exact_match, 'exactly' );
				}
				if ( ! empty( $starts_with ) ) {
					$where[] = $prepare_where( $starts_with, 'starts' );
				}
				if ( ! empty( $contains ) ) {
					$where[] = $prepare_where( $contains, 'contains' );
				}

				$where = implode( ' OR ', $where );

				$append = "
					$wpdb->posts.ID IN (
						SELECT DISTINCT
							tr.object_id
						FROM $wpdb->terms AS t
						INNER JOIN $wpdb->term_taxonomy AS tt ON t.term_id = tt.term_id
						INNER JOIN $wpdb->term_relationships AS tr ON tt.term_taxonomy_id = tr.term_taxonomy_id
						WHERE 1 = 1
							AND ($where)
					)
				";

				$append = empty( $append ) ? $append : "($append) OR ";

				$search = str_replace( ')))', '))))', $search );
				$search = str_replace( '(((', "($append(((", $search );
			}

			return $search;
		}

		/**
		 * Modifies JOIN clause of the WP_Query.
		 *
		 * @since 1.12.0
		 *
		 * @noinspection PhpMissingParamTypeInspection,PhpMissingReturnTypeInspection
		 *
		 * @param string   $join  - Join clause.
		 * @param WP_Query $query - WordPress query object.
		 *
		 * @return string
		 */
		public function wp_query_posts_join( $join, $query ) {
			global $wpdb;

			$is_search_by_products = $query->is_search
				&& 'product' === $query->get( 'post_type' )
				&& ! empty( $query->query_vars['search_terms'] );

			if ( ! $is_search_by_products ) {
				return $join;
			}

			if ( 'yes' === Settings::instance()->get( 'search_by_sku' ) ) {
				$join = $join . " LEFT JOIN $wpdb->posts AS redparts_sputnik_p1 ON $wpdb->posts.ID = redparts_sputnik_p1.post_parent AND redparts_sputnik_p1.post_type = 'product_variation'";
				$join = $join . " LEFT JOIN $wpdb->postmeta AS redparts_sputnik_pm1 ON ($wpdb->posts.ID = redparts_sputnik_pm1.post_id OR redparts_sputnik_p1.ID = redparts_sputnik_pm1.post_id) AND redparts_sputnik_pm1.meta_key = '_sku'";
			}

			return $join;
		}

		/**
		 * Modifies search query.
		 *
		 * @deprecated since 1.12.0 and will be removed from version 2.0.0.
		 *
		 * @param WP_Query $query Query object.
		 */
		public function pre_get_posts( WP_Query $query ) { }

		/**
		 * Returns the replaced search query if it was modified using the pre_get_posts hook.
		 *
		 * @deprecated since 1.12.0 and will be removed from version 2.0.0.
		 *
		 * @param string $default Default search query.
		 *
		 * @return string
		 */
		public function get_search_query( string $default ): string {
			return $default;
		}

		/**
		 * Output search suggestions.
		 */
		public function ajax_search_suggestions() {
			if (
				! isset( $_POST['nonce'] ) ||
				! wp_verify_nonce( sanitize_key( wp_unslash( $_POST['nonce'] ) ), 'redparts_sputnik_search_suggestions' )
			) {
				wp_send_json_error();
				return;
			}

			WPML::switch_ajax_language();

			if ( ! isset( $_POST['id'] ) || ! isset( $_POST['s'] ) ) {
				wp_send_json_error();
				return;
			}

			$form_id = sanitize_text_field( wp_unslash( $_POST['id'] ) );
			$s       = sanitize_text_field( wp_unslash( $_POST['s'] ) );

			$query_args = array(
				's'              => $s,
				'sentence'       => false,
				'post_type'      => 'product',
				'post_status'    => 'publish',
				'posts_per_page' => 20,
				// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
				'tax_query'      => array(
					'relation' => 'AND',
				),
			);

			$link_args = array(
				'post_type' => 'product',
				's'         => $s,
			);

			$query_args['tax_query'][] = array(
				'taxonomy' => 'product_visibility',
				'field'    => 'name',
				'terms'    => array( 'exclude-from-search' ),
				'operator' => 'NOT IN',
			);

			if ( ! empty( $_POST['redparts_taxonomy'] ) && ! empty( $_POST['redparts_taxonomy_value'] ) ) {
				$taxonomy       = sanitize_key( wp_unslash( $_POST['redparts_taxonomy'] ) );
				$taxonomy_value = sanitize_text_field( wp_unslash( $_POST['redparts_taxonomy_value'] ) );
				$taxonomy_terms = array();

				if ( Vehicles::instance()->get_attribute_slug() === $taxonomy ) {
					if ( preg_match( '#^[0-9]+$#', $taxonomy_value ) ) {
						$vehicle = Vehicles::instance()->get_vehicle_by_id( absint( $taxonomy_value ) );
					} else {
						$vehicle = Vehicles::instance()->get_vehicle_by_instance_id( $taxonomy_value );
					}

					if ( $vehicle ) {
						$taxonomy_terms   = is_array( $vehicle['slug'] ) ? $vehicle['slug'] : array( $vehicle['slug'] );
						$taxonomy_terms[] = Vehicles::instance()->get_all_term_slug();

						$query_arg_name  = Vehicles::instance()->get_attribute_filter_name();
						$query_arg_value = Vehicles::instance()->get_attribute_filter_value( $vehicle );

						$link_args[ $query_arg_name ] = $query_arg_value;
					}
				} else {
					$taxonomy_terms = array( sanitize_key( $taxonomy_value ) );

					$link_args[ $taxonomy ] = $taxonomy_value;
				}

				// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
				$query_args['tax_query'][] = array(
					'taxonomy' => $taxonomy,
					'field'    => 'slug',
					'terms'    => $taxonomy_terms,
				);
			}

			$query = new WP_Query( $query_args );

			foreach ( $query->posts as $product ) {
				$query->setup_postdata( $product );
				?>
				<a
					href="<?php echo esc_url( get_the_permalink( $product ) ); ?>"
					class="th-suggestions__item th-suggestions__item--product th-suggestions__product"
					tabindex="-1"
					role="option"
					id="<?php echo esc_attr( $form_id . '-suggestions-option-product-' . $product->ID ); ?>"
					data-s="<?php echo esc_attr( wp_kses( get_the_title( $product ), array() ) ); ?>"
				>
					<div class="th-suggestions__product-image">
						<?php echo wp_kses( woocommerce_get_product_thumbnail( array( 44, 44 ) ), 'redparts_sputnik_image' ); ?>
					</div>
					<div class="th-suggestions__product-info">
						<div class="th-suggestions__product-name">
							<?php
							echo wp_kses_post(
								preg_replace(
									'/' . preg_quote( $s, '/' ) . '/i',
									'<strong>$0</strong>',
									get_the_title( $product )
								)
							);
							?>
						</div>
						<div class="th-suggestions__product-price th-price">
							<?php woocommerce_template_loop_price(); ?>
						</div>
					</div>
				</a>
				<?php
			}
			$query->reset_postdata();

			if ( $query->found_posts > $query->post_count ) {
				$link = add_query_arg( $link_args, home_url( '/' ) );

				?>
				<a
					href="<?php echo esc_url( $link ); ?>"
					class="th-suggestions__item th-suggestions__item--all-results"
					tabindex="-1"
					role="option"
					id="<?php echo esc_attr( $form_id . '-suggestions-option-all-results' ); ?>"
					data-s="<?php echo esc_attr( $s ); ?>"
				>
					<?php echo esc_html__( 'See all results', 'redparts-sputnik' ); ?>
				</a>
				<?php
			}

			wp_die();
		}
	}
}
