<?php
/**
 * Block products columns widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Widget;
use WP_Query;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Block_Products_Columns' ) ) {
	/**
	 * Class Block_Products_Columns
	 */
	class Block_Products_Columns extends Widget {
		/**
		 * Indicates whether to display the widget title or not.
		 *
		 * @var bool
		 */
		protected $display_title = false;

		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_block_products_columns',
				'description'                 => esc_html_x( 'Products Columns.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_block_products_columns',
				esc_html_x( 'RedParts: Block Products Columns', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			if ( ! class_exists( 'WooCommerce' ) ) {
				return;
			}

			$columns = array();

			if ( isset( $instance['columns'] ) && is_array( $instance['columns'] ) ) {
				$columns = array_values( $instance['columns'] );
			}

			$columns = array_map(
				function ( $column ) {
					if ( ! is_array( $column ) ) {
						$column = array();
					}

					if ( isset( $column['group_order'] ) && is_numeric( $column['group_order'] ) ) {
						$column['group_order'] = (int) $column['group_order'];
					} else {
						$column['group_order'] = 0;
					}

					return $column;
				},
				$columns
			);

			usort(
				$columns,
				function ( $a, $b ) {
					return $a['group_order'] - $b['group_order'];
				}
			);

			if ( empty( $columns ) ) {
				return;
			}

			$product_visibility_term_ids = wc_get_product_visibility_term_ids();

			?>
			<!-- .th-block-product-columns -->
			<div class="th-block th-block-products-columns">
				<div class="th-container">
					<div class="th-row">
						<?php foreach ( $columns as $column ) : ?>
							<?php
							$title      = isset( $column['title'] ) && is_string( $column['title'] ) ? $column['title'] : '';
							$number     = isset( $column['number'] ) ? absint( $column['number'] ) : 3;
							$show       = isset( $column['show'] ) ? sanitize_title( $column['show'] ) : 'all';
							$products   = isset( $column['products'] ) ? wp_parse_slug_list( $column['products'] ) : array();
							$categories = isset( $column['categories'] ) ? wp_parse_slug_list( $column['categories'] ) : array();
							$orderby    = isset( $column['orderby'] ) ? sanitize_title( $column['orderby'] ) : 'date';
							$order      = isset( $column['order'] ) ? sanitize_title( $column['order'] ) : 'desc';

							$query_args = array(
								'posts_per_page' => $number,
								'post_status'    => 'publish',
								'post_type'      => 'product',
								'no_found_rows'  => 1,
								'order'          => $order,
								// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
								'tax_query'      => array(
									'relation' => 'AND',
								),
							);

							if ( 'yes' === get_option( 'woocommerce_hide_out_of_stock_items' ) ) {
								$query_args['tax_query'][] = array(
									array(
										'taxonomy' => 'product_visibility',
										'field'    => 'term_taxonomy_id',
										'terms'    => $product_visibility_term_ids['outofstock'],
										'operator' => 'NOT IN',
									),
								);
							}

							if ( ! empty( $products ) ) {
								$query_args['post_name__in'] = $products;
							}

							if ( ! empty( $categories ) ) {
								$query_args['tax_query'][] = array(
									'taxonomy' => 'product_cat',
									'field'    => 'slug',
									'terms'    => $categories,
								);
							}

							switch ( $show ) {
								case 'featured':
									$query_args['tax_query'][] = array(
										'taxonomy' => 'product_visibility',
										'field'    => 'term_taxonomy_id',
										'terms'    => $product_visibility_term_ids['featured'],
									);
									break;
								case 'onsale':
									$product_ids_on_sale    = wc_get_product_ids_on_sale();
									$product_ids_on_sale[]  = 0;
									$query_args['post__in'] = $product_ids_on_sale;
									break;
							}

							switch ( $orderby ) {
								case 'price':
									// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
									$query_args['meta_key'] = '_price';
									$query_args['orderby']  = 'meta_value_num';
									break;
								case 'sales':
									// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
									$query_args['meta_key'] = 'total_sales';
									$query_args['orderby']  = 'meta_value_num';
									break;
								case 'rating':
									// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
									$query_args['meta_key'] = '_wc_average_rating';
									$query_args['orderby']  = 'meta_value_num';
									break;
								case 'product_slugs':
									$query_args['orderby'] = 'post_name__in';
									break;
								default:
									$query_args['orderby'] = 'date';
							}

							$products = new WP_Query( $query_args );

							?>
							<div class="th-col-<?php echo esc_attr( 12 / count( $columns ) ); ?>">
								<?php if ( $title ) : ?>
									<div class="th-block-products-columns__title">
										<?php echo esc_html( $title ); ?>
									</div>
								<?php endif; ?>

								<ul class="th-block-products-columns__list">
									<?php while ( $products->have_posts() ) : ?>
										<?php
										$products->the_post();

										$exclude_elements = array(
											'featured_attributes',
											'description',
											'addtocart',
											'footer_wishlist',
											'footer_compare',
											'action_wishlist',
											'action_compare',
											'meta',
											'compatibility',
										);

										wc_set_loop_prop( 'redparts_class', 'th-block-products-columns__list-item' );
										wc_set_loop_prop( 'redparts_product_card_exclude', $exclude_elements );

										wc_get_template_part( 'content', 'product' );
										?>
									<?php endwhile; ?>
								</ul>
							</div>
							<?php
							$products->reset_postdata();
							wc_reset_loop();
							?>
						<?php endforeach; ?>
					</div>
				</div>
			</div>
			<?php
		}

		/**
		 * Returns form fields.
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'        => 'columns',
					'select_key' => 'current_column',
					'label'      => esc_html__( 'Products columns', 'redparts-sputnik' ),
					'type'       => 'tabs',
					'max'        => 3,
					'fields'     => array(
						array(
							'key'          => 'title',
							'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'     => 'group_order',
							'label'   => esc_html__( 'Group order', 'redparts-sputnik' ),
							'type'    => 'number',
							'default' => 0,
						),
						array(
							'key'     => 'number',
							'label'   => esc_html__( 'Number of products to show', 'redparts-sputnik' ),
							'type'    => 'number',
							'default' => 0,
						),
						array(
							'key'     => 'show',
							'label'   => esc_html__( 'Show', 'redparts-sputnik' ),
							'type'    => 'select',
							'default' => 'all',
							'options' => array(
								array(
									'key'   => 'all',
									'label' => esc_html__( 'All products', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'featured',
									'label' => esc_html__( 'Featured products', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'onsale',
									'label' => esc_html__( 'On-sale products', 'redparts-sputnik' ),
								),
							),
						),
						array(
							'key'   => 'products',
							'label' => esc_html__( 'Products (comma separated slugs)', 'redparts-sputnik' ),
							'type'  => 'text',
						),
						array(
							'key'   => 'categories',
							'label' => esc_html__( 'Categories (comma separated slugs)', 'redparts-sputnik' ),
							'type'  => 'text',
						),
						array(
							'key'     => 'orderby',
							'label'   => esc_html__( 'Order by', 'redparts-sputnik' ),
							'type'    => 'select',
							'default' => 'date',
							'options' => array(
								array(
									'key'   => 'date',
									'label' => esc_html__( 'Date', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'price',
									'label' => esc_html__( 'Price', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'sales',
									'label' => esc_html__( 'Sales', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'rating',
									'label' => esc_html__( 'Average rating', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'product_slugs',
									'label' => esc_html__( 'Product slugs', 'redparts-sputnik' ),
								),
							),
						),
						array(
							'key'     => 'order',
							'label'   => esc_html__( 'Order', 'redparts-sputnik' ),
							'type'    => 'select',
							'default' => 'desc',
							'options' => array(
								array(
									'key'   => 'desc',
									'label' => esc_html__( 'DESC', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'asc',
									'label' => esc_html__( 'ASC', 'redparts-sputnik' ),
								),
							),
						),
					),
					'l10n'       => array(
						'add'    => esc_html__( 'Add Column', 'redparts-sputnik' ),
						'remove' => esc_html__( 'Remove Column', 'redparts-sputnik' ),
						/* translators: %s: Column number. */
						'title'  => esc_html__( 'Column %s', 'redparts-sputnik' ),
					),
				),
			);
		}

		/**
		 * Enqueue admin scripts.
		 */
		public function enqueue_admin_scripts() {
			wp_enqueue_style( 'redparts-sputnik-admin' );
			wp_enqueue_script( 'redparts-sputnik-admin' );
		}
	}
}
