<?php
/**
 * Main plugin class.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Plugin' ) ) {
	/**
	 * Class Plugin
	 */
	class Plugin extends Singleton {
		const VERSION = '1.15.0';

		/**
		 * Array of plugin modules.
		 *
		 * @var array
		 */
		protected $modules = array(
			'RedParts\Sputnik\Settings',
			'RedParts\Sputnik\Database',
			'RedParts\Sputnik\Megamenu\Megamenu',
			'RedParts\Sputnik\Swatches\Swatches',
			'RedParts\Sputnik\Swatches\Label',
			'RedParts\Sputnik\Swatches\Color',
			'RedParts\Sputnik\Share_Buttons',
			'RedParts\Sputnik\Social_Links',
			'RedParts\Sputnik\AJAX',
			'RedParts\Sputnik\Attributes_Finder',
			'RedParts\Sputnik\Admin',
			'RedParts\Sputnik\Brands',
			'RedParts\Sputnik\Compare',
			'RedParts\Sputnik\Featured_Attributes',
			'RedParts\Sputnik\Filters',
			'RedParts\Sputnik\Garage',
			'RedParts\Sputnik\Product_Card',
			'RedParts\Sputnik\Quickview',
			'RedParts\Sputnik\Vehicles',
			'RedParts\Sputnik\Wishlist',
			'RedParts\Sputnik\Search',
			'RedParts\Sputnik\WPML\WPML',
			'RedParts\Sputnik\Lazy_Select',
		);

		/**
		 * Array of plugin widgets.
		 *
		 * @var array
		 */
		protected $widgets = array(
			'Block_About_Us',
			'Block_Attributes_Finder',
			'Block_Banner',
			'Block_Brands',
			'Block_Categories',
			'Block_Features',
			'Block_Finder',
			'Block_Posts_Carousel',
			'Block_Products_Carousel',
			'Block_Products_Columns',
			'Block_Reviews',
			'Block_Sale',
			'Block_Slideshow',
			'Block_Space',
			'Block_Teammates',
			'Block_Zone',
			'Filter_Vehicle',
			'Filters',
			'Footer_Contacts',
			'Footer_Links',
			'Footer_Newsletter',
			'Recent_Comments',
			'Recent_Posts',
		);

		/**
		 * Basic initialization.
		 */
		public function init() {
			add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
			add_action( 'wp_footer', array( $this, 'enqueue_footer_scripts' ) );
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
			add_action( 'elementor/editor/before_enqueue_scripts', array( $this, 'enqueue_elementor_scripts' ), 5 );
			add_action( 'widgets_init', array( $this, 'init_widgets' ) );
			add_action( 'plugins_loaded', array( $this, 'init_plugin' ) );

			// Initialize modules.
			foreach ( $this->modules as $module ) {
				$module::instance()->init();
			}
		}

		/**
		 * Initialize plugin.
		 */
		public function init_plugin() {
			// Load text domain.
			load_plugin_textdomain( 'redparts-sputnik', false, plugin_basename( $this->path( 'languages' ) ) );

			add_image_size( 'redparts-post-card-thumbnail', 560, 560, false );
		}

		/**
		 * Initialize widgets.
		 */
		public function init_widgets() {
			// Initialize widgets.
			foreach ( $this->widgets as $name ) {
				register_widget( '\RedParts\Sputnik\Widgets\\' . $name );
			}
		}

		/**
		 * Returns path of specified file.
		 *
		 * @param string $path The path to the file relative to the root directory of the plugin.
		 * @return string
		 */
		public function path( string $path ): string {
			return realpath( plugin_dir_path( __FILE__ ) . '../' . $path );
		}

		/**
		 * Returns URL of specified file.
		 *
		 * @param string $path The path to the file relative to the root directory of the plugin.
		 * @return string
		 */
		public function url( string $path ): string {
			return plugin_dir_url( __DIR__ ) . $path;
		}

		/**
		 * Enqueue scripts.
		 */
		public function enqueue_scripts() {
			$direction = is_rtl() ? '-rtl' : '-ltr';

			// Register scripts.
			wp_register_script(
				'owl-carousel',
				$this->url( 'assets/vendor/owl-carousel/owl.carousel.js' ),
				array( 'jquery' ),
				'2.3.4',
				true
			);
			wp_register_script(
				'redparts-sputnik-garage',
				$this->url( 'assets/js/garage.js' ),
				array( 'jquery', 'selectWoo' ),
				self::VERSION,
				true
			);
			wp_register_script(
				'redparts-sputnik-common',
				$this->url( 'assets/js/common.js' ),
				array(),
				self::VERSION,
				true
			);
			wp_register_script(
				'redparts-sputnik-widget-block-slideshow',
				$this->url( 'assets/js/widget-block-slideshow.js' ),
				array( 'jquery', 'owl-carousel', 'redparts-sputnik-common' ),
				self::VERSION,
				true
			);
			wp_register_script(
				'redparts-sputnik-widget-block-products-carousel',
				$this->url( 'assets/js/widget-block-products-carousel.js' ),
				array( 'jquery', 'owl-carousel', 'redparts-sputnik-common' ),
				self::VERSION,
				true
			);
			wp_register_script(
				'redparts-sputnik-widget-block-posts-carousel',
				$this->url( 'assets/js/widget-block-posts-carousel.js' ),
				array( 'jquery', 'owl-carousel', 'redparts-sputnik-common' ),
				self::VERSION,
				true
			);
			wp_register_script(
				'redparts-sputnik-widget-block-sale',
				$this->url( 'assets/js/widget-block-sale.js' ),
				array( 'jquery', 'owl-carousel', 'redparts-sputnik-common' ),
				self::VERSION,
				true
			);
			wp_register_script(
				'redparts-sputnik-widget-block-zone',
				$this->url( 'assets/js/widget-block-zone.js' ),
				array( 'jquery', 'owl-carousel', 'redparts-sputnik-common' ),
				self::VERSION,
				true
			);
			wp_register_script(
				'redparts-sputnik-widget-block-teammates',
				$this->url( 'assets/js/widget-block-teammates.js' ),
				array( 'jquery', 'owl-carousel', 'redparts-sputnik-common' ),
				self::VERSION,
				true
			);
			wp_register_script(
				'redparts-sputnik-widget-block-reviews',
				$this->url( 'assets/js/widget-block-reviews.js' ),
				array( 'jquery', 'owl-carousel', 'redparts-sputnik-common' ),
				self::VERSION,
				true
			);
			wp_register_script(
				'redparts-sputnik-widget-block-finder',
				$this->url( 'assets/js/widget-block-finder.js' ),
				array( 'jquery', 'redparts-sputnik-common', 'redparts-sputnik-garage' ),
				self::VERSION,
				true
			);
			wp_register_script(
				'redparts-sputnik-widget-block-attributes-finder',
				$this->url( 'assets/js/widget-block-attributes-finder.js' ),
				array( 'jquery', 'redparts-sputnik-common' ),
				self::VERSION,
				true
			);

			// Register styles.
			wp_register_style(
				'font-awesome',
				$this->url( 'assets/vendor/fontawesome/css/font-awesome.min.css' ),
				array(),
				'4.7.0'
			);
			wp_register_style(
				'redparts-sputnik-style',
				$this->url( 'assets/css/style' . $direction . '.min.css' ),
				array( 'font-awesome', 'select2' ),
				self::VERSION
			);
			wp_register_style(
				'redparts-sputnik-inline',
				false,
				array(),
				self::VERSION
			);

			// Enqueue styles.
			wp_enqueue_style( 'redparts-sputnik-style' );

			do_action( 'redparts_sputnik_enqueue_scripts' );
		}

		/**
		 * Enqueue footer scripts.
		 */
		public function enqueue_footer_scripts() {
			// Enqueue styles.
			wp_enqueue_style( 'redparts-sputnik-inline' );
		}

		/**
		 * Enqueue scripts on admin pages.
		 */
		public function enqueue_admin_scripts() {
			$direction = is_rtl() ? '-rtl' : '-ltr';

			// Register scripts.
			wp_register_script(
				'papaparse',
				$this->url( 'assets/vendor/papaparse/papaparse.min.js' ),
				array(),
				'5.3.0',
				true
			);
			wp_register_script(
				'js-base64',
				$this->url( 'assets/vendor/js-base64/base64.js' ),
				array(),
				'3.6.0',
				true
			);
			wp_register_script(
				'redparts-sputnik-admin',
				$this->url( 'assets/js/admin.js' ),
				array( 'underscore', 'jquery', 'wp-i18n', 'wp-color-picker' ),
				self::VERSION,
				true
			);
			wp_localize_script(
				'redparts-sputnik-admin',
				'redPartsSputnikVars',
				array(
					'selectImage' => esc_html__( 'Select Image', 'redparts-sputnik' ),
					'column'      => esc_html__( 'Column', 'redparts-sputnik' ),
				)
			);
			wp_register_script(
				'redparts-sputnik-megamenu',
				$this->url( 'assets/js/megamenu.js' ),
				array( 'redparts-sputnik-admin', 'jquery', 'nav-menu' ),
				self::VERSION,
				true
			);
			wp_register_script(
				'redparts-sputnik-lazy-select',
				$this->url( 'assets/js/lazy-select.js' ),
				array( 'redparts-sputnik-admin', 'jquery', 'selectWoo' ),
				self::VERSION,
				true
			);
			// Register styles.
			wp_register_style(
				'redparts-sputnik-admin',
				$this->url( 'assets/css/admin' . $direction . '.min.css' ),
				array( 'wp-color-picker' ),
				self::VERSION
			);

			// Enqueue.
			wp_enqueue_style( 'redparts-sputnik-admin' );
			wp_enqueue_script( 'redparts-sputnik-admin' );

			$screen    = get_current_screen();
			$screen_id = $screen ? $screen->id : '';

			if ( class_exists( 'WooCommerce' ) && in_array( $screen_id, wc_get_screen_ids(), true ) ) {
				wp_enqueue_script( 'redparts-sputnik-lazy-select' );
			};
		}

		/**
		 * Enqueue elementor scripts.
		 */
		public function enqueue_elementor_scripts() {
			$this->enqueue_admin_scripts();
		}

		/**
		 * Returns widget classes.
		 *
		 * @since 1.6.0
		 *
		 * @return string[]
		 */
		public function get_widget_classes(): array {
			return array_map(
				function ( $widget_name ) {
					return '\RedParts\Sputnik\Widgets\\' . $widget_name;
				},
				$this->widgets
			);
		}
	}
}
