<?php
/**
 * This file contains code related to the color swatches.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Swatches;

use RedParts\Sputnik\Singleton;
use RedParts\Sputnik\Colors;
use RedParts\Sputnik\WPML\WPML;
use WP_Term;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Swatches\Color' ) ) {
	/**
	 * Class Color
	 */
	class Color extends Singleton {
		const TERM_META_NAME = 'redparts_sputnik_color';

		/**
		 * Initialization.
		 */
		public function init() {
			add_action( 'admin_init', array( $this, 'admin_init' ) );
			add_filter( 'redparts_sputnik_swatches_types', array( $this, 'get_attribute_types' ) );
			add_action( 'redparts_sputnik_swatches_control_template', array( $this, 'control_template' ), 10, 3 );
		}

		/**
		 * Deferred initialization.
		 *
		 * @noinspection PhpUnused
		 */
		public function admin_init() {
			if ( ! class_exists( 'WooCommerce' ) ) {
				return;
			}

			$attribute_taxonomies = wc_get_attribute_taxonomies();

			foreach ( $attribute_taxonomies as $attribute_taxonomy ) {
				if ( 'color' !== $attribute_taxonomy->attribute_type ) {
					continue;
				}

				$taxonomy_name = wc_attribute_taxonomy_name( $attribute_taxonomy->attribute_name );

				add_filter( 'manage_edit-' . $taxonomy_name . '_columns', array( $this, 'terms_table_columns' ) );
				add_filter( 'manage_' . $taxonomy_name . '_custom_column', array( $this, 'terms_table_color_column_content' ), 10, 3 );
				add_action( $taxonomy_name . '_add_form_fields', array( $this, 'term_add_form_color_field' ) );
				add_action( $taxonomy_name . '_edit_form_fields', array( $this, 'term_edit_form_color_field' ) );
				add_action( 'create_' . $taxonomy_name, array( $this, 'save_term_color' ) );
				add_action( 'edit_' . $taxonomy_name, array( $this, 'save_term_color' ) );
			}
		}

		/**
		 * Returns an associative array of attribute types.
		 *
		 * @noinspection PhpUnused
		 *
		 * @param array $types An associative array of attribute types.
		 *
		 * @return array
		 */
		public function get_attribute_types( $types = array() ): array {
			return array_merge( $types, array( 'color' => esc_html__( 'Color', 'redparts-sputnik' ) ) );
		}

		/**
		 * Returns columns of terms table.
		 *
		 * @noinspection PhpUnused
		 *
		 * @param array $columns Columns array.
		 *
		 * @return array
		 */
		public function terms_table_columns( array $columns ): array {
			$cb_index = array_search( 'cb', array_keys( $columns ), true );

			if ( false === $cb_index ) {
				$columns[ self::TERM_META_NAME ] = esc_attr_x( 'Color', 'Admin', 'redparts-sputnik' );
			} else {
				$columns = array_slice( $columns, 0, $cb_index + 1, true )
					+ array( self::TERM_META_NAME => esc_attr_x( 'Color', 'Admin', 'redparts-sputnik' ) )
					+ array_slice( $columns, $cb_index + 1, null, true );
			}

			return $columns;
		}

		/**
		 * Outputs content of color column.
		 *
		 * @noinspection PhpUnused
		 *
		 * @param string $content     Column content.
		 * @param string $column_name Column name.
		 * @param int    $term_id     Term ID.
		 *
		 * @return string
		 */
		public function terms_table_color_column_content( string $content, string $column_name, int $term_id ): string {
			if ( self::TERM_META_NAME !== $column_name ) {
				return $content;
			}

			$color = sanitize_hex_color( $this->get_term_color( absint( $term_id ) ) );

			if ( ! $color ) {
				return $content;
			}

			/**
			 * From WordPress Theme Coding Requirements:
			 * "Dynamic styling must be added via wp_add_inline_style() with the exception of adding a background image to an element."
			 * This case also falls under this exception. Because we may not need to override the background color from another stylesheet.
			 *
			 * @link https://help.author.envato.com/hc/en-us/articles/360000479946
			 */
			return '<div class="redparts-sputnik-color-swatch" style="' . esc_attr( 'background-color: ' . $color . ';' ) . '"></div>';
		}

		/**
		 * Adds a color field to the term add form.
		 */
		public function term_add_form_color_field() {
			?>
			<div class="form-field term-redparts-color-value-wrap">
				<label for="redparts-sputnik-term-color"><?php esc_html_e( 'Color', 'redparts-sputnik' ); ?></label>
				<input
					id="redparts-sputnik-term-color"
					class="redparts-color-picker"
					name="<?php echo esc_attr( self::TERM_META_NAME ); ?>"
					type="text"
					value="#000"
				>
			</div>
			<?php
		}

		/**
		 * Adds a color field to the term edit form.
		 *
		 * @param WP_Term $term Term object.
		 */
		public function term_edit_form_color_field( WP_Term $term ) {
			$color = $this->get_term_color( $term );

			if ( ! is_string( $color ) || ! preg_match( '#\#([A-Fa-f0-9]{3}){1,2}#', $color ) ) {
				$color = '#000';
			}

			?>
			<tr class="form-field term-redparts-color-value-wrap">
				<th scope="row"><label for="redparts-sputnik-term-color"><?php esc_html_e( 'Color Value', 'redparts-sputnik' ); ?></label>
				</th>
				<td>
					<input
						id="redparts-sputnik-term-color"
						class="redparts-color-picker"
						name="<?php echo esc_attr( self::TERM_META_NAME ); ?>"
						type="text"
						value="<?php echo esc_attr( $color ); ?>"
					>
				</td>
			</tr>
			<?php
		}

		/**
		 * Saves term color.
		 *
		 * @param int $term_id Term ID.
		 */
		public function save_term_color( int $term_id ) {
			if ( ! isset( $_POST[ self::TERM_META_NAME ] ) ) {
				return;
			}
			if ( ! current_user_can( 'edit_term', $term_id ) ) {
				return;
			}
			if ( ! isset( $_POST['_wpnonce'] ) && ! isset( $_POST['_wpnonce_add-tag'] ) ) {
				return;
			}

			// Sanitization here would be redundant.
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			if (
				! wp_verify_nonce( wp_unslash( $_POST['_wpnonce'] ), 'update-tag_' . $term_id )
				&& ! wp_verify_nonce( wp_unslash( $_POST['_wpnonce_add-tag'] ), 'add-tag' )
			) {
				return;
			}
			// phpcs:enable

			$this->update_term_color( $term_id, sanitize_hex_color( wp_unslash( $_POST[ self::TERM_META_NAME ] ) ) );
		}

		/**
		 * Outputs control template.
		 *
		 * @param array     $args   Args array.
		 * @param string    $type   Attribute type.
		 * @param WP_Term[] $values Option values.
		 */
		public function control_template( array $args, string $type, array $values ) {
			if ( 'color' !== $type ) {
				return;
			}

			?>
			<div class="th-input-radio-color">
				<div class="th-input-radio-color__list">
					<?php foreach ( $values as $value ) : ?>
						<?php
						$color_value = sanitize_hex_color( $this->get_term_color( $value ) );
						$color_value = '' === $color_value ? '#000' : $color_value;
						$color_type  = Colors::type( $color_value );

						$item_classes = array(
							'th-input-radio-color__item',
							'th-input-radio-color__item--' . $color_type,
						);

						?>
						<label
							class="<?php echo esc_attr( implode( ' ', $item_classes ) ); ?>"
							data-toggle="tooltip"
							<?php
							/**
							 * From WordPress Theme Coding Requirements:
							 * "Dynamic styling must be added via wp_add_inline_style() with the exception of adding a background image to
							 * an element." This case also falls under this exception. Because we may not need to override the CSS color
							 * from another stylesheet.
							 *
							 * @link https://help.author.envato.com/hc/en-us/articles/360000479946
							 */
							?>
							style="color: <?php echo esc_attr( $color_value ); ?>"
							title="<?php echo esc_attr( $value->name ); ?>"
						>
							<input
								type="radio"
								class="th-input-radio-color__input"
								name="redparts_fake_option_<?php echo esc_attr( $args['attribute'] ); ?>"
								value="<?php echo esc_attr( $value->slug ); ?>"
							>
							<span class="th-input-radio-color__circle"></span>
							<span class="th-input-radio-color__stick"></span>
						</label>
					<?php endforeach; ?>
				</div>
			</div>
			<?php
		}

		/**
		 * Returns term color.
		 *
		 * @since 1.6.0
		 *
		 * @param int|WP_Term $term Term ID or WP_Term object.
		 * @return mixed
		 */
		public function get_term_color( $term ) {
			return get_term_meta( WPML::get_original_term_id( $term ), self::TERM_META_NAME, true );
		}

		/**
		 * Updates term color.
		 *
		 * @since 1.6.0
		 *
		 * @param int|WP_Term $term  Term ID or WP_Term object.
		 * @param string      $color Color.
		 */
		public function update_term_color( $term, string $color ) {
			update_term_meta( WPML::get_original_term_id( $term ), self::TERM_META_NAME, $color );
		}
	}
}
