<?php
/**
 * Block attributes finder widget.
 *
 * @package RedParts\Sputnik
 * @since 1.7.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Attributes_Finder;
use RedParts\Sputnik\Settings;
use RedParts\Sputnik\Widget;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Block_Attributes_Finder' ) ) {
	/**
	 * Class Block_Attributes_Finder
	 *
	 * @since 1.7.0
	 */
	class Block_Attributes_Finder extends Widget {
		/**
		 * Indicates whether to display the widget title or not.
		 *
		 * @since 1.7.0
		 *
		 * @var bool
		 */
		protected $display_title = false;

		/**
		 * Constructor.
		 *
		 * @since 1.7.0
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_block_attributes_finder',
				'description'                 => esc_html_x( 'Displays a part finder block. Unlike the usual "Block Finder", the data for this block is taken from the product attributes and not from the vehicle fields.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_block_attributes_finder',
				esc_html_x( 'RedParts: Block Attributes Finder', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Echoes the widget content.
		 *
		 * @since 1.7.0
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		public function widget( $args, $instance ) {
			if ( 'yes' !== Settings::instance()->get( 'autoparts_features' ) ) {
				return;
			}

			parent::widget( $args, $instance );
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @since 1.7.0
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			if ( ! class_exists( 'WooCommerce' ) ) {
				return;
			}

			$title    = $instance['title'] ?? '';
			$subtitle = $instance['subtitle'] ?? '';
			$button   = $instance['button'] ?? '';
			$image    = isset( $instance['image'] ) ? absint( $instance['image'] ) : '';
			$strict   = isset( $instance['strict'] ) && 'yes' === $instance['strict'];
			$fields   = isset( $instance['fields'] ) ? (array) $instance['fields'] : array();

			$fields = array_map(
				function ( $field_instance ) {
					if ( ! is_array( $field_instance ) ) {
						return null;
					}

					$slug = $field_instance['slug'] ?? '';

					if ( empty( $slug ) || ! taxonomy_is_product_attribute( $slug ) ) {
						return null;
					}

					$default_label = wc_attribute_label( $slug );
					// translators: %s attribute label.
					$default_placeholder = sprintf( esc_html__( 'Select %s', 'redparts-sputnik' ), $default_label );

					$label       = ! empty( $field_instance['label'] ) ? $field_instance['label'] : $default_label;
					$placeholder = ! empty( $field_instance['placeholder'] ) ? $field_instance['placeholder'] : $default_placeholder;
					$order       = $field_instance['order'] ?? 0;

					return array(
						'slug'        => $slug,
						'label'       => $label,
						'placeholder' => $placeholder,
						'order'       => $order,
					);
				},
				$fields
			);
			$fields = array_filter(
				$fields,
				function ( $filed ) {
					return null !== $filed;
				}
			);

			uasort(
				$fields,
				function( $a, $b ) {
					$a = intval( $a['order'] );
					$b = intval( $b['order'] );

					if ( $a === $b ) {
						return 0;
					}

					return ( $a < $b ) ? -1 : 1;
				}
			);

			// Reset keys.
			$fields = array_values( $fields );

			$field_slugs = array_map(
				function( $field ) {
					return $field['slug'];
				},
				$fields
			);

			?>
			<div
				class="th-block th-block-finder th-block-finder--type--attributes"
				data-ajax-url="<?php echo esc_url( apply_filters( 'redparts_sputnik_ajax_url', '' ) ); ?>"
				data-nonce="<?php echo esc_attr( wp_create_nonce( 'redparts_sputnik_attributes_finder' ) ); ?>"
				data-strict="<?php echo esc_attr( $strict ? 1 : 0 ); ?>"
			>
				<div class="th-decor th-decor--type--bottom th-block-finder__decor">
					<div class="th-decor__body">
						<div class="th-decor__start"></div>
						<div class="th-decor__end"></div>
						<div class="th-decor__center"></div>
					</div>
				</div>
				<div
					class="th-block-finder__image"
					style="background-image: url('<?php echo esc_url( wp_get_attachment_image_url( $image, 'full' ) ); ?>');"
				></div>
				<div class="th-block-finder__body th-container th-container--max--xl">
					<?php if ( ! empty( $title ) || ! empty( $subtitle ) ) : ?>
						<div class="th-block-finder__header">
							<?php if ( ! empty( $title ) ) : ?>
								<div class="th-block-finder__title">
									<?php echo esc_html( $title ); ?>
								</div>
							<?php endif; ?>
							<?php if ( ! empty( $subtitle ) ) : ?>
								<div class="th-block-finder__subtitle">
									<?php echo esc_html( $subtitle ); ?>
								</div>
							<?php endif; ?>
						</div>
					<?php endif; ?>
					<form
						class="th-block-finder__form"
						action="<?php echo esc_url( get_permalink( wc_get_page_id( 'shop' ) ) ); ?>"
						method="get"
					>
						<div class="th-block-finder__controls">
							<?php foreach ( $fields as $idx => $field ) : ?>
								<?php
								$options      = 0 === $idx ? Attributes_Finder::instance()->get_options( $strict ? $field_slugs : array( $field['slug'] ) ) : array();
								$item_classes = array( 'th-block-finder__control' );

								if ( 0 !== $idx ) {
									$item_classes[] = 'th-block-finder__control--disabled';
								}
								?>
								<div
									class="<?php echo esc_attr( implode( ' ', $item_classes ) ); ?>"
									data-label="<?php echo esc_attr( $field['label'] ); ?>"
								>
									<select
										class="th-block-finder__control-input"
										name="<?php echo esc_attr( 'filter_' . wc_attribute_taxonomy_slug( $field['slug'] ) ); ?>"
										aria-label="<?php echo esc_attr( $field['label'] ); ?>"
										<?php disabled( 0 !== $idx ); ?>
									>
										<option value=""><?php echo esc_html( $field['placeholder'] ); ?></option>
										<?php foreach ( $options as $option ) : ?>
											<option value="<?php echo esc_attr( $option['slug'] ); ?>"><?php echo esc_html( $option['name'] ); ?></option>
										<?php endforeach; ?>
									</select>
									<div class="th-block-finder__control-loader"></div>
								</div>
							<?php endforeach; ?>
						</div>
						<button
							class="th-block-finder__button"
							type="submit"
						>
							<?php
							if ( empty( $button ) ) {
								echo esc_html__( 'Search', 'redparts-sputnik' );
							} else {
								echo esc_html( $button );
							}
							?>
						</button>
					</form>
				</div>
			</div>
			<?php
		}

		/**
		 * Returns form fields.
		 *
		 * @since 1.7.0
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'          => 'title',
					'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => $this->is_elementor_widget(),
				),
				array(
					'key'          => 'subtitle',
					'label'        => esc_html__( 'Subtitle', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => true,
				),
				array(
					'key'          => 'button',
					'label'        => esc_html__( 'Button Text', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => true,
				),
				array(
					'key'   => 'image',
					'label' => esc_html__( 'Image', 'redparts-sputnik' ),
					'type'  => 'image',
				),
				array(
					'key'     => 'strict',
					'label'   => esc_html__( 'Strict (show only possible combinations)', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'yes',
							'label' => esc_html__( 'Yes', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'no',
							'label' => esc_html__( 'No', 'redparts-sputnik' ),
						),
					),
					'default' => 'no',
				),
				array(
					'key'        => 'fields',
					'select_key' => 'current_field',
					'label'      => esc_html__( 'Fields', 'redparts-sputnik' ),
					'type'       => 'tabs',
					'max'        => Attributes_Finder::MAX_PROCESSED_FIELDS,
					'fields'     => array(
						array(
							'key'          => 'slug',
							'label'        => esc_html__( 'Product attribute slug', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'label',
							'label'        => esc_html__( 'Label', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'placeholder',
							'label'        => esc_html__( 'Placeholder', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'     => 'order',
							'label'   => esc_html__( 'Order', 'redparts-sputnik' ),
							'type'    => 'number',
							'default' => 0,
						),
					),
					'l10n'       => array(
						'add'    => esc_html__( 'Add Field', 'redparts-sputnik' ),
						'remove' => esc_html__( 'Remove Field', 'redparts-sputnik' ),
						/* translators: %s: Group number. */
						'title'  => esc_html__( 'Field %s', 'redparts-sputnik' ),
					),
				),
			);
		}

		/**
		 * Enqueue frontend scripts.
		 *
		 * @since 1.7.0
		 */
		public function enqueue_frontend_scripts() {
			wp_enqueue_script( 'redparts-sputnik-widget-block-attributes-finder' );
		}

		/**
		 * Enqueue admin scripts.
		 *
		 * @since 1.7.0
		 */
		public function enqueue_admin_scripts() {
			wp_enqueue_media();
		}
	}
}
