<?php
/**
 * Block sale widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Widget;
use WP_Query;
use DateTime;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Block_Sale' ) ) {
	/**
	 * Class Block_Sale
	 */
	class Block_Sale extends Widget {
		/**
		 * Indicates whether to display the widget title or not.
		 *
		 * @var bool
		 */
		protected $display_title = false;

		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_block_sale',
				'description'                 => esc_html_x( 'Displays products on sale.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_block_sale',
				esc_html_x( 'RedParts: Block Sale', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			if ( ! class_exists( 'WooCommerce' ) ) {
				return;
			}

			$title      = isset( $instance['title'] ) ? $instance['title'] : '';
			$subtitle   = isset( $instance['subtitle'] ) ? $instance['subtitle'] : '';
			$end_date   = isset( $instance['end_date'] ) ? $instance['end_date'] : '';
			$link_title = isset( $instance['link_title'] ) ? $instance['link_title'] : '';
			$link_url   = isset( $instance['link_url'] ) ? $instance['link_url'] : '';
			$image      = isset( $instance['image'] ) ? $instance['image'] : '';

			$mobile_grid_columns  = max( 1, min( 2, abs( $instance['mobile_grid_columns'] ?? '1' ) ) );
			$autoplay             = ( $instance['autoplay'] ?? 'no' ) === 'yes';
			$autoplay_hover_pause = ( $instance['autoplay_hover_pause'] ?? 'yes' ) === 'yes';
			$autoplay_timeout     = absint( $instance['autoplay_timeout'] ?? '5000' );
			$number               = isset( $instance['number'] ) ? absint( $instance['number'] ) : 0;
			$show                 = isset( $instance['show'] ) ? sanitize_title( $instance['show'] ) : 'all';
			$products             = isset( $instance['products'] ) ? wp_parse_slug_list( $instance['products'] ) : '';
			$categories           = isset( $instance['categories'] ) ? wp_parse_slug_list( $instance['categories'] ) : '';
			$orderby              = isset( $instance['orderby'] ) ? sanitize_title( $instance['orderby'] ) : 'date';
			$order                = isset( $instance['order'] ) ? sanitize_title( $instance['order'] ) : 'desc';

			if ( ! empty( $end_date ) ) {
				$end_date = DateTime::createFromFormat( 'Y-m-d H:i:s', $end_date );
				$end_date = false === $end_date ? '' : $end_date->getTimestamp();
			}

			$product_visibility_term_ids = wc_get_product_visibility_term_ids();

			$query_args = array(
				'posts_per_page' => $number,
				'post_status'    => 'publish',
				'post_type'      => 'product',
				'no_found_rows'  => 1,
				'order'          => $order,
				// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
				'tax_query'      => array(
					'relation' => 'AND',
				),
			);

			if ( 'yes' === get_option( 'woocommerce_hide_out_of_stock_items' ) ) {
				$query_args['tax_query'][] = array(
					array(
						'taxonomy' => 'product_visibility',
						'field'    => 'term_taxonomy_id',
						'terms'    => $product_visibility_term_ids['outofstock'],
						'operator' => 'NOT IN',
					),
				);
			}

			if ( ! empty( $products ) ) {
				$query_args['post_name__in'] = $products;
			}

			if ( ! empty( $categories ) ) {
				$query_args['tax_query'][] = array(
					'taxonomy' => 'product_cat',
					'field'    => 'slug',
					'terms'    => $categories,
				);
			}

			switch ( $show ) {
				case 'featured':
					$query_args['tax_query'][] = array(
						'taxonomy' => 'product_visibility',
						'field'    => 'term_taxonomy_id',
						'terms'    => $product_visibility_term_ids['featured'],
					);
					break;
				case 'onsale':
					$product_ids_on_sale    = wc_get_product_ids_on_sale();
					$product_ids_on_sale[]  = 0;
					$query_args['post__in'] = $product_ids_on_sale;
					break;
			}

			switch ( $orderby ) {
				case 'price':
					// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
					$query_args['meta_key'] = '_price';
					$query_args['orderby']  = 'meta_value_num';
					break;
				case 'sales':
					// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
					$query_args['meta_key'] = 'total_sales';
					$query_args['orderby']  = 'meta_value_num';
					break;
				case 'rating':
					// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
					$query_args['meta_key'] = '_wc_average_rating';
					$query_args['orderby']  = 'meta_value_num';
					break;
				case 'product_slugs':
					$query_args['orderby'] = 'post_name__in';
					break;
				default:
					$query_args['orderby'] = 'date';
			}

			$products = new WP_Query( $query_args );

			?>
			<!-- .th-block-sale -->
			<div
				class="block th-block-sale"
				data-autoplay="<?php echo esc_attr( $autoplay ? 1 : 0 ); ?>"
				data-autoplay-hover-pause="<?php echo esc_attr( $autoplay_hover_pause ? 1 : 0 ); ?>"
				data-autoplay-timeout="<?php echo esc_attr( $autoplay_timeout ); ?>"
				data-mobile-grid-columns="<?php echo esc_attr( $mobile_grid_columns ); ?>"
			>
				<div class="th-block-sale__content">
					<div class="th-block-sale__header">
						<?php if ( ! empty( $title ) ) : ?>
							<div class="th-block-sale__title">
								<?php echo esc_html( $title ); ?>
							</div>
						<?php endif; ?>
						<?php if ( ! empty( $subtitle ) ) : ?>
							<div class="th-block-sale__subtitle">
								<?php echo esc_html( $subtitle ); ?>
							</div>
						<?php endif; ?>

						<?php if ( '' !== $end_date ) : ?>
							<div class="th-block-sale__timer">
								<div class="th-timer" data-seconds-left="<?php echo esc_attr( max( 0, $end_date - time() ) ); ?>">
									<div class="th-timer__part">
										<div class="th-timer__part-value th-timer__part-value--days"></div>
										<div class="th-timer__part-label">
											<?php echo esc_html__( 'Days', 'redparts-sputnik' ); ?>
										</div>
									</div>
									<div class="th-timer__dots"></div>
									<div class="th-timer__part">
										<div class="th-timer__part-value th-timer__part-value--hours"></div>
										<div class="th-timer__part-label">
											<?php echo esc_html__( 'Hrs', 'redparts-sputnik' ); ?>
										</div>
									</div>
									<div class="th-timer__dots"></div>
									<div class="th-timer__part">
										<div class="th-timer__part-value th-timer__part-value--minutes"></div>
										<div class="th-timer__part-label">
											<?php echo esc_html__( 'Mins', 'redparts-sputnik' ); ?>
										</div>
									</div>
									<div class="th-timer__dots"></div>
									<div class="th-timer__part">
										<div class="th-timer__part-value th-timer__part-value--seconds"></div>
										<div class="th-timer__part-label">
											<?php echo esc_html__( 'Secs', 'redparts-sputnik' ); ?>
										</div>
									</div>
								</div>
							</div>
						<?php endif; ?>

						<div class="th-block-sale__controls">
							<div class="th-arrow th-arrow--direction--prev th-block-sale__arrow th-block-sale__arrow--prev">
								<button class="th-arrow__button" type="button">
									<svg xmlns="http://www.w3.org/2000/svg" width="7" height="11">
										<path
											d="M6.7.3c-.4-.4-.9-.4-1.3 0L0 5.5l5.4 5.2c.4.4.9.3 1.3 0 .4-.4.4-1 0-1.3l-4-3.9 4-3.9c.4-.4.4-1 0-1.3z"></path>
									</svg>
								</button>
							</div>

							<?php if ( ! empty( $link_title ) ) : ?>
								<div class="th-block-sale__link">
									<a href="<?php echo esc_url( $link_url ); ?>">
										<?php echo esc_html( $link_title ); ?>
									</a>
								</div>
							<?php endif; ?>

							<div class="th-arrow th-arrow--direction--next th-block-sale__arrow th-block-sale__arrow--next">
								<button class="th-arrow__button" type="button">
									<svg xmlns="http://www.w3.org/2000/svg" width="7" height="11">
										<path
											d="M.3 10.7c.4.4.9.4 1.3 0L7 5.5 1.6.3C1.2-.1.7 0 .3.3c-.4.4-.4 1 0 1.3l4 3.9-4 3.9c-.4.4-.4 1 0 1.3z"></path>
									</svg>
								</button>
							</div>
							<div class="th-decor th-decor--type--center th-block-sale__header-decor">
								<div class="th-decor__body">
									<div class="th-decor__start"></div>
									<div class="th-decor__end"></div>
									<div class="th-decor__center"></div>
								</div>
							</div>
						</div>
					</div>
					<div class="th-block-sale__body">
						<div class="th-decor th-decor--type--bottom th-block-sale__body-decor">
							<div class="th-decor__body">
								<div class="th-decor__start"></div>
								<div class="th-decor__end"></div>
								<div class="th-decor__center"></div>
							</div>
						</div>

						<?php if ( ! empty( $image ) ) : ?>
							<div class="th-block-sale__image">
								<?php echo wp_get_attachment_image( $image, 'full', false, array( 'role' => 'presentation' ) ); ?>
							</div>
						<?php endif; ?>

						<div class="th-container">
							<div class="th-block-sale__carousel">
								<div class="owl-carousel">
									<?php while ( $products->have_posts() ) : ?>
										<ul class="th-block-sale__item">
											<?php
											$products->the_post();

											$exclude_elements = array(
												'featured_attributes',
												'description',
												'footer_wishlist',
												'footer_compare',
											);

											wc_set_loop_prop( 'redparts_product_card_exclude', $exclude_elements );

											wc_get_template_part( 'content', 'product' );
											?>
										</ul>
									<?php endwhile; ?>
								</div>
							</div>
						</div>
					</div>
				</div>
			</div>
			<!-- .th-block-sale / end -->
			<?php

			wp_reset_postdata();
			wc_reset_loop();
		}

		/**
		 * Returns form fields.
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'          => 'title',
					'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => $this->is_elementor_widget(),
				),
				array(
					'key'          => 'subtitle',
					'label'        => esc_html__( 'Subtitle', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => true,
				),
				array(
					'key'         => 'end_date',
					'label'       => esc_html__( 'End date', 'redparts-sputnik' ),
					'type'        => 'text',
					'placeholder' => 'Y-m-d H:i:s',
				),
				array(
					'key'          => 'link_title',
					'label'        => esc_html__( 'Link title', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => true,
				),
				array(
					'key'          => 'link_url',
					'label'        => esc_html__( 'Link url', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => true,
				),
				array(
					'key'   => 'image',
					'label' => esc_html__( 'Image', 'redparts-sputnik' ),
					'type'  => 'image',
				),
				array(
					'key'     => 'mobile_grid_columns',
					'label'   => esc_html__( 'Mobile grid columns', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => '1',
							'label' => esc_html__( '1 column', 'redparts-sputnik' ),
						),
						array(
							'key'   => '2',
							'label' => esc_html__( '2 columns', 'redparts-sputnik' ),
						),
					),
					'default' => '1',
				),
				array(
					'key'     => 'autoplay',
					'label'   => esc_html__( 'Autoplay', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'yes',
							'label' => esc_html__( 'Yes', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'no',
							'label' => esc_html__( 'No', 'redparts-sputnik' ),
						),
					),
					'default' => 'no',
				),
				array(
					'key'     => 'autoplay_hover_pause',
					'label'   => esc_html__( 'Pause autoplay on hover', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'yes',
							'label' => esc_html__( 'Yes', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'no',
							'label' => esc_html__( 'No', 'redparts-sputnik' ),
						),
					),
					'default' => 'yes',
				),
				array(
					'key'     => 'autoplay_timeout',
					'label'   => esc_html__( 'Autoplay timeout', 'redparts-sputnik' ),
					'type'    => 'number',
					'default' => '5000',
				),
				array(
					'key'     => 'number',
					'label'   => esc_html__( 'Number of products to show', 'redparts-sputnik' ),
					'type'    => 'number',
					'default' => 0,
				),
				array(
					'key'     => 'show',
					'label'   => esc_html__( 'Show', 'redparts-sputnik' ),
					'type'    => 'select',
					'default' => 'all',
					'options' => array(
						array(
							'key'   => 'all',
							'label' => esc_html__( 'All products', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'featured',
							'label' => esc_html__( 'Featured products', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'onsale',
							'label' => esc_html__( 'On-sale products', 'redparts-sputnik' ),
						),
					),
				),
				array(
					'key'   => 'products',
					'label' => esc_html__( 'Products (comma separated slugs)', 'redparts-sputnik' ),
					'type'  => 'text',
				),
				array(
					'key'   => 'categories',
					'label' => esc_html__( 'Categories (comma separated slugs)', 'redparts-sputnik' ),
					'type'  => 'text',
				),
				array(
					'key'     => 'orderby',
					'label'   => esc_html__( 'Order by', 'redparts-sputnik' ),
					'type'    => 'select',
					'default' => 'date',
					'options' => array(
						array(
							'key'   => 'date',
							'label' => esc_html__( 'Date', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'price',
							'label' => esc_html__( 'Price', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'sales',
							'label' => esc_html__( 'Sales', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'rating',
							'label' => esc_html__( 'Average rating', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'product_slugs',
							'label' => esc_html__( 'Product slugs', 'redparts-sputnik' ),
						),
					),
				),
				array(
					'key'     => 'order',
					'label'   => esc_html__( 'Order', 'redparts-sputnik' ),
					'type'    => 'select',
					'default' => 'desc',
					'options' => array(
						array(
							'key'   => 'desc',
							'label' => esc_html__( 'DESC', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'asc',
							'label' => esc_html__( 'ASC', 'redparts-sputnik' ),
						),
					),
				),
			);
		}

		/**
		 * Enqueue admin scripts.
		 */
		public function enqueue_admin_scripts() {
			wp_enqueue_style( 'redparts-sputnik-admin' );
			wp_enqueue_script( 'redparts-sputnik-admin' );
		}

		/**
		 * Enqueue frontend scripts.
		 */
		public function enqueue_frontend_scripts() {
			wp_enqueue_script( 'redparts-sputnik-widget-block-sale' );
		}
	}
}
