<?php
/**
 * WPML widgets module.
 *
 * @package RedParts\Sputnik
 * @since 1.6.0
 **/

namespace RedParts\Sputnik\WPML;

use RedParts\Sputnik\Plugin;
use WPML_PB_String;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\WPML\Widgets_Module' ) ) {
	/**
	 * Class Widgets_Module
	 *
	 * This class does not implement the IWPML_Page_Builders_Module interface because we are passing $element by
	 * reference in the update method.
	 *
	 * @since 1.6.0
	 */
	class Widgets_Module {
		/**
		 * Returns WPML strings.
		 *
		 * @noinspection PhpMissingParamTypeInspection
		 * @since 1.6.0
		 *
		 * @param string|int       $node_id Elementor node ID.
		 * @param mixed            $element Element data.
		 * @param WPML_PB_String[] $strings Array of WPML strings.
		 *
		 * @return WPML_PB_String[]
		 */
		public function get( $node_id, $element, $strings ): array {
			$widget_fields = $this->get_widget_fields( $element );

			$callback = function( WPML_PB_String $each_string ) use ( &$strings ) {
				$strings[] = $each_string;
			};

			$this->walker(
				$callback,
				$element['settings']['wp'],
				$widget_fields,
				array( $element['widgetType'], $node_id )
			);

			return $strings;
		}

		/**
		 * Sets the translated value for the specified WPML string.
		 *
		 * @since 1.6.0
		 *
		 * @param string|int     $node_id Elementor node ID.
		 * @param mixed          $element Element data.
		 * @param WPML_PB_String $string  Current WPML string.
		 */
		public function update( $node_id, &$element, WPML_PB_String $string ) {
			$widget_fields = $this->get_widget_fields( $element );

			$callback = function( WPML_PB_String $each_string, &$value ) use ( $string ) {
				if ( $each_string->get_name() === $string->get_name() ) {
					$value = $string->get_value();
				}
			};

			$this->walker(
				$callback,
				$element['settings']['wp'],
				$widget_fields,
				array( $element['widgetType'], $node_id )
			);
		}

		/**
		 * Walks by widget fields and calls callback function for all translatable fields.
		 *
		 * @since 1.6.0
		 *
		 * @param callback $callback   Callback function.
		 * @param array    $instance   Widget settings.
		 * @param array    $fields     Array of widget fields.
		 * @param array    $path       Path to the current widget fields.
		 * @param array    $label_path Path to the current widget fields from labels.
		 */
		protected function walker(
			callable $callback,
			array &$instance,
			array $fields,
			array $path = array(),
			array $label_path = array()
		) {
			foreach ( $fields as $field ) {
				$field_key  = $field['key'];
				$field_type = $field['type'];

				switch ( $field_type ) {
					case 'text':
					case 'textarea':
						$value_exists    = ! empty( $instance[ $field_key ] );
						$is_translatable = ! empty( $field['translatable'] ) && $field['translatable'];

						if ( $value_exists && $is_translatable ) {
							$string_path  = array_merge( $path, array( $field_key ) );
							$string_value = $instance[ $field_key ];
							$string_name  = implode( '-', $string_path );
							$string       = new WPML_PB_String(
								$string_value,
								$string_name,
								implode( ' / ', array_merge( $label_path, array( $field['label'] ) ) ),
								'text' === $field_type ? 'LINE' : 'AREA'
							);

							$callback(
								$string,
								$instance[ $field_key ]
							);
						}

						break;
					case 'tabs':
						$value_exists = ! empty( $instance[ $field_key ] );
						$is_array     = is_array( $instance[ $field_key ] );

						if ( $value_exists && $is_array ) {
							foreach ( $instance[ $field_key ] as $tab_idx => &$tab_instance ) {
								if ( ! is_array( $tab_instance ) ) {
									continue;
								}

								$tab_fields = $field['fields'];
								$tab_path   = array_merge( $path, array( $field_key, $tab_idx ) );

								$this->walker(
									$callback,
									$tab_instance,
									$tab_fields,
									$tab_path,
									array_merge( $label_path, array( $field['label'], $tab_idx ) )
								);
							}
						}

						break;
				}
			}
		}

		/**
		 * Returns widget fields by specified element.
		 *
		 * @since 1.6.0
		 *
		 * @param mixed $element Element data.
		 *
		 * @return array
		 */
		protected function get_widget_fields( $element ): array {
			static $cache = null;

			if ( null === $cache ) {
				$cache = array();

				foreach ( Plugin::instance()->get_widget_classes() as $widget_class ) {
					$widget         = new $widget_class();
					$widget->number = 'REPLACE_TO_ID';

					if ( method_exists( $widget, 'form_fields' ) ) {
						$cache[ 'wp-widget-' . $widget->id_base ] = $widget->form_fields();
					}
				}
			}

			$widget_type = $element['widgetType'];

			if ( isset( $cache[ $widget_type ] ) ) {
				return $cache[ $widget_type ];
			}

			return array();
		}
	}
}
