<?php namespace WooPlugins\Permalinks;

use WooPlugins\Permalinks\Core\AdminNotifier;
use WooPlugins\Permalinks\Core\FileManager;
use WooPlugins\Permalinks\Core\ServiceContainerTrait;
use WooPlugins\Permalinks\Services\FlushService;
use WooPlugins\Permalinks\Services\PermalinkService;
use WooPlugins\Permalinks\Services\ProductSpecificPermalinks;
use WooPlugins\Permalinks\Services\RedirectService;
use WooPlugins\Permalinks\Settings\Settings;

/**
 * Class PermalinksPlugin
 *
 * @package WooPlugins\Permalinks
 */
class PermalinksPlugin {

	use ServiceContainerTrait;

	const VERSION = '1.2.0';

	/**
	 * PermalinksPlugin constructor.
	 *
	 * @param string $mainFile
	 */
	public function __construct( $mainFile ) {
		FileManager::init( $mainFile, 'custom-permalinks-for-woocommerce' );

		add_action( 'plugins_loaded', array( $this, 'loadTextDomain' ) );
	}

	/**
	 * Run plugin part
	 */
	public function run() {
		$this->getContainer()->add( 'fileManager', FileManager::getInstance() );
		$this->getContainer()->add( 'adminNotifier', new AdminNotifier() );

		if ( $this->checkRequirements() ) {

			if ( get_transient( 'custom_permalinks_for_woocommerce_activated' ) ) {
				add_action( 'admin_notices', array( $this, 'showActivationMessage' ) );
			}

			$this->initServices();
		}
	}

	public function showActivationMessage() {
		$link = $this->getContainer()->getSettings()->getLink();
		$this->getContainer()->getFileManager()->includeTemplate( 'admin/activation-message.php', array( 'link' => $link ) );

		delete_transient( 'custom_permalinks_for_woocommerce_activated' );
	}

	public function initServices() {

		$this->getContainer()->add( 'settings', new Settings() );
		$this->getContainer()->add( 'permalinkService', new PermalinkService() );
		$this->getContainer()->add( 'redirectService', new RedirectService() );
		$this->getContainer()->add( 'flushService', new FlushService() );
		$this->getContainer()->add( 'productSpecificPermalinkService', new ProductSpecificPermalinks() );

		add_filter( 'plugin_action_links_' . plugin_basename( $this->getContainer()->getFileManager()->getMainFile() ), function ( $actions ) {
			$actions[] = '<a href="' . $this->getContainer()->getSettings()->getLink() . '">' . __( 'Settings', 'custom-permalinks-for-woocommerce' ) . '</a>';

			return $actions;
		}, 10, 4 );

		do_action( 'custom_permalinks/container/services_init' );
	}

	/**
	 * Load plugin translations
	 */
	public function loadTextDomain() {
		$name = $this->getContainer()->getFileManager()->getPluginName();
		load_plugin_textdomain( 'custom-permalinks-for-woocommerce', false, $name . '/languages/' );
	}

	public function activate() {

		set_transient( 'custom_permalinks_for_woocommerce_activated', true, 100 );

		flush_rewrite_rules();
	}

	public function deactivate() {
		flush_rewrite_rules();
	}

	public static function uninstall() {
		flush_rewrite_rules();
	}

	public function checkRequirements() {
		/* translators: %s = required plugin */
		$message = __( 'The Custom Permalinks for WooCommerce requires %s plugin to be active!', 'custom-permalinks-for-woocommerce' );

		$plugins = $this->getRequiredPluginsToBeActive();

		if ( count( $plugins ) ) {
			foreach ( $plugins as $plugin ) {
				$error = sprintf( $message, $plugin );
				$this->getContainer()->getAdminNotifier()->push( $error, AdminNotifier::ERROR, false );
			}

			return false;
		}

		return true;
	}

	public function getRequiredPluginsToBeActive() {

		$plugins = array();

		if ( ! function_exists( 'is_plugin_active' ) ) {
			include_once( ABSPATH . 'wp-admin/includes/plugin.php' );
		}

		if ( ! ( is_plugin_active( 'woocommerce/woocommerce.php' ) || is_plugin_active_for_network( 'woocommerce/woocommerce.php' ) ) ) {
			$plugins[] = 'WooCommerce';
		}

		return $plugins;
	}
}
