<?php namespace WooPlugins\Permalinks\Services;

use WooPlugins\Permalinks\Core\ServiceContainerTrait;
use WooPlugins\Permalinks\Settings\Settings;
use WP_Post;

/**
 * Class PermalinkService
 *
 * Main class to adjust posts' and categories' links
 *
 */
class PermalinkService {

	use ServiceContainerTrait;

	private $productBase;

	private $polyLangInstance;

	public function __construct() {

		add_filter( 'post_type_link', array( $this, 'adjustProductPermalink' ), - 99, 2 );
		add_filter( 'term_link', array( $this, 'adjustCategoryPermalink' ), - 99, 3 );

		add_filter( 'rewrite_rules_array', array( $this, 'adjustWPRewriteRules' ), 99 );

		// Polylang integration
		add_action( 'pll_init', function ( $polylang ) {
			$this->polyLangInstance = $polylang;
		} );
	}

	/**
	 * Adjust category permalink according to the permalink settings
	 *
	 * @param string $link
	 * @param object $term
	 * @param string $taxonomy
	 *
	 * @return string
	 */
	public function adjustCategoryPermalink( $link, $term, $taxonomy ) {

		if ( ! in_array( $taxonomy, array( 'product_cat' ) ) ) {
			return $link;
		}

		if ( ! $this->getContainer()->getSettings()->isPermalinkEnabledForCategory() && ! $this->getContainer()->getSettings()->isURLEndingEnabledForCategories() ) {
			return $link;
		}

		$urlEnding = false;

		if ( $this->getContainer()->getSettings()->isURLEndingEnabledForCategories() ) {
			$urlEnding = $this->getContainer()->getSettings()->getURLEnding();
			$urlEnding = $urlEnding ? $urlEnding : false;
		}

		$permalinkType = $this->getContainer()->getSettings()->get( 'category_permalink', 'default' );

		$path = $this->buildCategoryPath( $link, $term, $permalinkType, $urlEnding );

		if ( ! $this->getContainer()->getSettings()->isPermalinkEnabledForCategory() ) {
			return $path;
		}

		return $urlEnding ? home_url( $path ) : home_url( user_trailingslashit( $path ) );
	}

	/**
	 * Replace product permalink according to settings
	 *
	 * @param string $permalink
	 * @param WP_Post $post
	 *
	 * @return string
	 */
	public function adjustProductPermalink( $permalink, $post ) {

		if ( 'product' !== $post->post_type || ! get_option( 'permalink_structure' ) ) {
			return $permalink;
		}

		if ( $this->getContainer()->getProductSpecificPermalinkService()->isProductSpecificPermalinkEnabled( $post->ID ) ) {
			$permalink = str_replace( trim( $this->getProductBase(), '/' ), trim( $this->getProductBase( $post->ID ), '/' ), $permalink );
		}

		if ( $this->getContainer()->getSettings()->get( 'product_identifier', 'default' ) === 'sku' ) {
			$permalink = $this->useSKUForPermalink( $permalink, $post->ID );
		}

		if ( $this->getContainer()->getSettings()->isURLEndingEnabledForProducts() ) {

			$urlEnding = $this->getContainer()->getSettings()->getURLEnding();

			if ( $urlEnding ) {
				$permalink = rtrim( $permalink, '/' );
				$permalink .= $urlEnding;
			}
		}

		if ( ! $this->getContainer()->getSettings()->isPermalinkEnabledForProducts( $post->ID ) ) {
			return $permalink;
		}

		$productBase = $this->getProductBase( $post->ID );

		if ( strpos( $productBase, '%product_cat%' ) !== false ) {
			$productBase = str_replace( '%product_cat%', '', $productBase );
		}

		$productBase = '/' . trim( $productBase, '/' ) . '/';

		$link = str_replace( $productBase, '/', $permalink );

		$link = $this->addPostParentLink( $link, $post, $this->getContainer()->getSettings()->getProductPermalink( $post->ID ) );

		return $link;
	}

	/**
	 * Adjust rewrite rules for WP
	 *
	 * @param $rules
	 *
	 * @return array
	 */
	public function adjustWPRewriteRules( $rules ) {

		if ( ! $this->getContainer()->getSettings()->isPermalinkEnabledForCategory() && ! $this->getContainer()->getSettings()->isURLEndingEnabledForCategories() ) {
			return $rules;
		}

		wp_cache_flush();

		global $wp_rewrite;

		$feed = '(' . trim( implode( '|', $wp_rewrite->feeds ) ) . ')';

		$customRules = array();

		/**
		 * Remove WPML filters while getting terms, to get all languages
		 */
		if ( isset( $GLOBALS['sitepress'] ) ) {

			$sitepress = $GLOBALS['sitepress'];

			$has_get_terms_args_filter = remove_filter( 'get_terms_args', array(
				$sitepress,
				'get_terms_args_filter'
			) );
			$has_get_term_filter       = remove_filter( 'get_term', array( $sitepress, 'get_term_adjust_id' ), 1 );
			$has_terms_clauses_filter  = remove_filter( 'terms_clauses', array( $sitepress, 'terms_clauses' ) );
		}

		$taxonomy = 'product_cat';

		$terms = get_categories( array(
			'taxonomy'   => $taxonomy,
			'hide_empty' => false,
		) );

		$permalinkType = $this->getContainer()->getSettings()->get( 'category_permalink' );
		$urlEnding     = false;

		if ( $this->getContainer()->getSettings()->isURLEndingEnabledForCategories() ) {
			$urlEnding = $this->getContainer()->getSettings()->getURLEnding();
			$urlEnding = $urlEnding ? $urlEnding : false;
		}

		foreach ( $terms as $term ) {
			$slug = $this->buildCategoryPath( '', $term, $permalinkType, $urlEnding );

			$customRules["$slug/?$"]                                             = 'index.php?' . $taxonomy . '=' . $term->slug;
			$customRules["$slug/embed/?$"]                                       = 'index.php?' . $taxonomy . '=' . $term->slug . '&embed=true';
			$customRules["$slug/{$wp_rewrite->feed_base}/{$feed}/?$"]            = 'index.php?' . $taxonomy . '=' . $term->slug . '&feed=$matches[1]';
			$customRules["$slug/{$feed}/?$"]                                     = 'index.php?' . $taxonomy . '=' . $term->slug . '&feed=$matches[1]';
			$customRules["$slug/{$wp_rewrite->pagination_base}/?([0-9]{1,})/?$"] = 'index.php?' . $taxonomy . '=' . $term->slug . '&paged=$matches[1]';

			// Polylang compatibility
			$polylangURLslug = $this->getPolylangLangSlug();

			if ( $polylangURLslug ) {

				$slug = $polylangURLslug . $slug;

				$customRules["$slug/?$"]                                             = 'index.php?' . $taxonomy . '=' . $term->slug;
				$customRules["$slug/embed/?$"]                                       = 'index.php?' . $taxonomy . '=' . $term->slug . '&embed=true';
				$customRules["$slug/{$wp_rewrite->feed_base}/{$feed}/?$"]            = 'index.php?' . $taxonomy . '=' . $term->slug . '&feed=$matches[1]';
				$customRules["$slug/{$feed}/?$"]                                     = 'index.php?' . $taxonomy . '=' . $term->slug . '&feed=$matches[1]';
				$customRules["$slug/{$wp_rewrite->pagination_base}/?([0-9]{1,})/?$"] = 'index.php?' . $taxonomy . '=' . $term->slug . '&paged=$matches[1]';
			}
		}

		/**
		 * Register WPML filters back
		 */
		if ( isset( $sitepress ) ) {

			if ( ! empty( $has_terms_clauses_filter ) ) {
				add_filter( 'terms_clauses', array( $sitepress, 'terms_clauses' ), 10, 3 );
			}

			if ( ! empty( $has_get_term_filter ) ) {
				add_filter( 'get_term', array( $sitepress, 'get_term_adjust_id' ), 1, 1 );
			}
			if ( ! empty( $has_get_terms_args_filter ) ) {
				add_filter( 'get_terms_args', array( $sitepress, 'get_terms_args_filter' ), 10, 2 );
			}
		}

		return $customRules + $rules;
	}

	private function getPolylangLangSlug() {
		if ( ! empty( $this->polyLangInstance ) ) {
			global $wp_rewrite;

			$languages = $this->polyLangInstance->model->get_languages_list( array( 'fields' => 'slug' ) );

			if ( $this->polyLangInstance->options['hide_default'] ) {
				$languages = array_diff( $languages, array( $this->polyLangInstance->options['default_lang'] ) );
			}

			if ( ! empty( $languages ) ) {
				return $wp_rewrite->root . ( $this->polyLangInstance->options['rewrite'] ? '' : 'language/' ) . '(' . implode( '|', $languages ) . ')/';
			}
		}

		return false;
	}

	private function getProductBase( $specificProduct = null ) {

		$productSpecificPermalink = $this->getContainer()->getProductSpecificPermalinkService()->getProductSpecificPermalinkOption( $specificProduct );

		if ( ProductSpecificPermalinks::DEFAULT_VALUE !== $productSpecificPermalink ) {

			if ( 'default' === $productSpecificPermalink ) {
				return '/product/';
			} else if ( 'product_slug' === $productSpecificPermalink ) {
				return Settings::PERMALINK_WC_PRODUCT;
			} else if ( in_array( $productSpecificPermalink, array( 'category_slug', 'full' ) ) ) {
				return Settings::PERMALINK_WC_PRODUCT_CAT;
			} else {
				return '';
			}
		}

		if ( is_null( $this->productBase ) ) {
			$permalinkStructure = wc_get_permalink_structure();
			$this->productBase  = $permalinkStructure['product_rewrite_slug'];
		}

		return $this->productBase;
	}

	private function addPostParentLink( $permalink, $post, $type ) {
		if ( false === strpos( $permalink, '%product_cat%' ) ) {
			return $permalink;
		}

		$term = $this->getProductCategory( $post );

		if ( $term ) {
			$slug      = $this->buildCategoryPath( $permalink, $term, $type );
			$permalink = str_replace( '%product_cat%', $slug, $permalink );
		}

		return $permalink;

	}

	private function buildCategoryPath( $permalink, $term, $type, $urlEnding = false ) {

		$slug = urldecode( $term->slug );

		if ( 'full' === $type ) {

			if ( $term->parent ) {
				$ancestors = get_ancestors( $term->term_id, 'product_cat' );
				foreach ( $ancestors as $ancestor ) {
					$ancestor_object = get_term( $ancestor, 'product_cat' );
					$slug            = urldecode( $ancestor_object->slug ) . '/' . $slug;
				}
			}

			return $urlEnding ? $slug . $urlEnding : $slug;

		} else if ( 'category_slug' === $type ) {
			return $urlEnding ? $slug . $urlEnding : $slug;
		} else if ( 'default' === $type && $urlEnding ) {

			if ( $term->parent ) {
				$ancestors = get_ancestors( $term->term_id, 'product_cat' );
				foreach ( $ancestors as $ancestor ) {
					$ancestor_object = get_term( $ancestor, 'product_cat' );
					$slug            = urldecode( $ancestor_object->slug ) . '/' . $slug;
				}
			}

			if ( ! $permalink ) {
				$permalink = 'product-category/' . $slug;
			} else {
				$permalink = rtrim( $permalink, '/' );
			}

			return $permalink . $urlEnding;
		}

		return $permalink;
	}

	private function getProductCategory( $product ) {

		$term = null;

		if ( ! empty( $this->getContainer()->getSettings()->get( 'yoast_primary_category', 'yes' ) === 'yes' ) ) {
			$term = $this->getSeoPrimaryTerm( $product );
		}

		if ( ! ( $term instanceof \WP_Term ) ) {
			$term = $this->getWcPrimaryTerm( $product );
		}
		if ( $term instanceof \WP_Term ) {
			return $term;
		}

		return null;
	}

	private function getSeoPrimaryTerm( $product ) {
		if ( function_exists( 'yoast_get_primary_term_id' ) ) {

			$primaryTerm = yoast_get_primary_term_id( 'product_cat', $product->ID );

			return get_term( $primaryTerm );
		}

		return null;
	}

	private function getWcPrimaryTerm( $product ) {

		$terms = get_the_terms( $product->ID, 'product_cat' );

		if ( empty( $terms ) ) {
			return null;
		}
		if ( function_exists( 'wp_list_sort' ) ) {
			$terms = wp_list_sort( $terms, 'term_id' );
		} else {
			usort( $terms, '_usort_terms_by_ID' );
		}

		$category_object = $terms[0];
		$category_object = get_term( $category_object, 'product_cat' );

		return $category_object;
	}

	/**
	 * Use SKU for product permalink
	 *
	 * @param string $permalink
	 * @param integer $postID
	 *
	 * @return string
	 */
	protected function useSKUForPermalink( $permalink, $postID ) {
		$skuString = get_post_meta( $postID, '_sku', true );

		if ( '' != $skuString ) {
			return str_replace( basename( $permalink ), $skuString, $permalink );
		}

		return $permalink;
	}
}
