<?php namespace WooPlugins\Permalinks\Services;

use WooPlugins\Permalinks\Core\ServiceContainerTrait;
use WP_Post;
use WP_Term;

class RedirectService {

	use ServiceContainerTrait;

	public function __construct() {

		if ( $this->getContainer()->getSettings()->isProductSpecificPermalinksEnabled() || $this->getContainer()->getSettings()->isPermalinkEnabledForProducts() || $this->getContainer()->getSettings()->isURLEndingEnabledForProducts() ) {
			add_action( 'request', array( $this, 'replaceRequest' ), 11 );
		}

		if ( 'yes' === $this->getContainer()->getSettings()->get( 'canonicals', 'yes' ) ) {
			add_action( 'wp_head', array( $this, 'addCanonical' ) );
		}

		if ( 'yes' === $this->getContainer()->getSettings()->get( '301_redirects_enabled', 'yes' ) ) {
			add_action( 'template_redirect', array( $this, 'redirectCanonical' ) );
		}
	}

	/**
	 * Replace request if product found
	 *
	 * @param array $request
	 *
	 * @return array
	 */
	public function replaceRequest( $request ) {
		global $wp, $wpdb;

		$url = $wp->request;

		if ( ! empty( $this->getContainer()->getSettings()->getURLEnding() ) ) {
			$url = $this->removeURLEnding( $url, $this->getContainer()->getSettings()->getURLEnding() );
		}

		if ( ! empty( $url ) ) {
			$url = explode( '/', $url );

			$slug = array_pop( $url );

			$replace = array();

			if ( 'feed' === $slug ) {
				$replace['feed'] = $slug;
				$slug            = array_pop( $url );
			}

			if ( 'amp' === $slug ) {
				$replace['amp'] = $slug;
				$slug           = array_pop( $url );
			}

			$commentsPosition = strpos( $slug, 'comment-page-' );

			if ( 0 === $commentsPosition ) {
				$replace['cpage'] = substr( $slug, strlen( 'comment-page-' ) );
				$slug             = array_pop( $url );
			}

			if ( $this->getContainer()->getSettings()->get( 'product_identifier', 'default' ) === 'sku' ) {
				$slug = $this->getProductSlugBySKU( $slug );
			}

			$num = intval( $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(ID) as count_id FROM {$wpdb->posts} WHERE post_name = %s AND post_type = %s", array(
				$slug,
				'product'
			) ) ) );

			if ( $num > 0 ) {
				$replace['page']      = '';
				$replace['post_type'] = 'product';
				$replace['product']   = $slug;
				$replace['name']      = $slug;

				return $replace;
			}
		}

		return $request;
	}

	protected function removeURLEnding( $url, $ending ) {

		$length = mb_strlen( $ending );

		if ( 0 === $length ) {
			return true;
		}

		// Ends with
		if ( ( substr( $url, - $length ) === $ending ) ) {
			$url = str_replace( $ending, '', $url );
		}

		return $url;
	}

	public function addCanonical() {

		if ( ! defined( 'WPSEO_VERSION' ) && ( is_product() || is_product_category() ) ) {
			$canonical = $this->getCanonical();

			if ( ! empty( $canonical ) ) {
				echo '<link rel="canonical" href="' . esc_url( $canonical ) . '" />' . "\n";
			}
		}
	}

	public function redirectCanonical() {
		global $wp;

		if ( ! is_product() && ! is_product_category() && ! is_product_tag() ) {
			return;
		}

		$isAmp = ( false !== get_query_var( 'amp', false ) );

		if ( is_product() && $isAmp ) {
			return;
		}

		/**
		 * WP_Post
		 *
		 * @var WP_Post $post
		 */
		$post = get_queried_object();

		if ( $post instanceof WP_Post && 'publish' !== $post->post_status ) {
			return;
		}

		$canonical = $this->getCanonical( true );

		$location = null;

		if ( $canonical ) {
			$query    = empty( sanitize_text_field( $_SERVER['QUERY_STRING'] ) ) ? '' : '?' . sanitize_text_field( $_SERVER['QUERY_STRING'] );
			$location = $canonical . $query;
		}

		$currentUrl = esc_url( home_url( add_query_arg( array(), urldecode( $wp->request ) ) ) );

		if ( trim( urldecode( $canonical ), '/' ) == trim( $currentUrl, '/' ) ) {
			$location = null;
		}

		if ( $location ) {
			wp_safe_redirect( $location, 301 );
			die;
		}

	}

	private function getCanonical( $useCommentsPagination = false ) {
		global $wp_rewrite;

		$qo = get_queried_object();

		$canonical = null;

		if ( $qo instanceof WP_Term ) {
			$canonical = get_term_link( $qo );
			$paged     = get_query_var( 'paged' );

			if ( $paged > 1 ) {
				$canonical = trailingslashit( $canonical ) . trailingslashit( $wp_rewrite->pagination_base ) . $paged;
			}
		} elseif ( $qo instanceof WP_Post ) {
			$canonical = get_permalink( $qo );

			if ( $useCommentsPagination ) {
				$page = get_query_var( 'cpage' );

				if ( $page > 1 ) {
					$canonical = trailingslashit( $canonical ) . $wp_rewrite->comments_pagination_base . '-' . $page;
				}
			}

		}

		if ( $canonical ) {
			return user_trailingslashit( $canonical );
		}

		return false;
	}

	/**
	 * Find current slug by product SKU
	 *
	 * @param string $slug
	 *
	 * @return string
	 */
	protected function getProductSlugBySKU( $slug ) {

		global $wpdb;

		$sku = $wpdb->get_row( $wpdb->prepare( "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_value = %s AND meta_key = '_sku'", array( $slug ) ), ARRAY_A );

		if ( isset( $sku['post_id'] ) ) {
			$productSlug = get_post_field( 'post_name', $sku['post_id'] );

			if ( $productSlug ) {
				return $productSlug;
			}
		}

		return $slug;
	}

}
