<?php namespace WooPlugins\Permalinks\Settings;

use WooPlugins\Permalinks\Core\ServiceContainerTrait;
use WooPlugins\Permalinks\PermalinksPlugin;
use WooPlugins\Permalinks\Services\FlushService;
use WooPlugins\Permalinks\Services\ProductSpecificPermalinks;

class Settings {

	use ServiceContainerTrait;

	const SETTINGS_PREFIX = 'custom_permalinks_';

	const SETTINGS_PAGE = 'custom_permalinks_settings';

	const PERMALINK_STRUCTURE = '/%postname%/';

	const PERMALINK_WC_PRODUCT_CAT = '/shop/%product_cat%/';

	const PERMALINK_WC_PRODUCT = 'product';

	/**
	 * Array with the settings
	 *
	 * @var array
	 */
	private $settings;

	/**
	 * Settings constructor.
	 */
	public function __construct() {
		$this->getContainer()->add( PermalinkRadioOption::class, new PermalinkRadioOption() );
		$this->getContainer()->add( URLEndingOption::class, new URLEndingOption() );

		add_action( 'init', array( $this, 'initSettings' ) );

		add_filter( 'woocommerce_settings_tabs_' . self::SETTINGS_PAGE, array( $this, 'registerSettings' ) );
		add_filter( 'woocommerce_settings_tabs_array', array( $this, 'addSettingsTab' ), 50 );
		add_action( 'woocommerce_update_options_' . self::SETTINGS_PAGE, array( $this, 'updateSettings' ) );

		add_action( 'woocommerce_admin_settings_sanitize_option_' . self::SETTINGS_PREFIX . 'url_ending', array(
			$this,
			'sanitizeURLEnding'
		), 3, 10 );

		add_action( 'woocommerce_admin_settings_sanitize_option_' . self::SETTINGS_PREFIX . 'product_permalink', array(
			$this,
			'updateWCProductPermalinks'
		) );
		add_action( 'woocommerce_admin_settings_sanitize_option_' . self::SETTINGS_PREFIX . 'category_permalink', array(
			$this,
			'updateWCCategoryPermalinks'
		) );
	}

	public function updateWCCategoryPermalinks( $value ) {
		if ( 'default' !== $value && ! get_option( 'permalink_structure' ) ) {
			update_option( 'permalink_structure', self::PERMALINK_STRUCTURE );
		};

		return $value;
	}

	public function updateWCProductPermalinks( $value ) {

		if ( 'default' !== $value && ! get_option( 'permalink_structure' ) ) {
			update_option( 'permalink_structure', self::PERMALINK_STRUCTURE );
		};

		if ( 'product_slug' === $value ) {
			$wcPermalinks['product_base'] = self::PERMALINK_WC_PRODUCT;

			update_option( 'woocommerce_permalinks', $wcPermalinks );

		} else if ( in_array( $value, array( 'category_slug', 'full' ) ) ) {
			$wcPermalinks['product_base'] = self::PERMALINK_WC_PRODUCT_CAT;

			update_option( 'woocommerce_permalinks', $wcPermalinks );
		} else {
			$wcPermalinks['product_base'] = '';

			update_option( 'woocommerce_permalinks', $wcPermalinks );
		}


		return $value;
	}

	/**
	 * Handle updating settings
	 */
	public function updateSettings() {
		woocommerce_update_options( $this->settings );
	}

	public function getProductIdentifier() {
		return 'sku' === $this->get( 'product_identifier', 'default' ) ? 'SKU' : 'product-slug';
	}

	/**
	 * Init all settings
	 */
	public function initSettings() {

		$settings = array(
			'main_section' => array(
				'title' => __( 'Custom Permalinks', 'custom-permalinks-for-woocommerce' ),
				'desc'  => __( 'Choose what permalinks you wanna see for your products and categories, add URL ending, or change product slug in setting down below. Be aware, when you save changes to permalink settings, all URLs in your store will be re-created.',
					'custom-permalinks-for-woocommerce' ),
				'id'    => self::SETTINGS_PREFIX . 'main_section',
				'type'  => 'title',
			),
			array(
				'title'              => __( 'Product permalink', 'custom-permalinks-for-woocommerce' ),
				'id'                 => self::SETTINGS_PREFIX . 'product_permalink',
				'default'            => 'default',
				'type'               => 'custom_permalinks_radio',
				'product_identifier' => $this->getProductIdentifier(),
				'options'            => array(
					'default'       => array(
						'label'       => __( 'Use WooCommerce defaults', 'custom-permalinks-for-woocommerce' ),
						'description' => home_url( 'product/' ),
					),
					'product_slug'  => array(
						'label'       => __( 'Product slug alone', 'custom-permalinks-for-woocommerce' ),
						'description' => home_url( '/' ),
					),
					'category_slug' => array(
						'label'       => __( 'Product slug with category name', 'custom-permalinks-for-woocommerce' ),
						'description' => home_url( 'product-category/' ),
					),
					'full'          => array(
						'label'       => __( 'Full path', 'custom-permalinks-for-woocommerce' ),
						'description' => home_url( 'parent-category/category/' ),
					),
				),
			),
			array(
				'title'   => __( 'Category permalink', 'custom-permalinks-for-woocommerce' ),
				'id'      => self::SETTINGS_PREFIX . 'category_permalink',
				'default' => 'default',
				'type'    => 'custom_permalinks_radio',
				'options' => array(
					'default'       => array(
						'label'       => __( 'Use defaults', 'custom-permalinks-for-woocommerce' ),
						'description' => home_url( 'product-category/category' ),
					),
					'category_slug' => array(
						'label'       => __( 'Category slug alone', 'custom-permalinks-for-woocommerce' ),
						'description' => home_url( 'category' ),
					),
					'full'          => array(
						'label'       => __( 'Full path', 'custom-permalinks-for-woocommerce' ),
						'description' => home_url( 'parent-category/category' ),
					),
				),
			),
			array(
				'title'    => __( 'Product slug', 'custom-permalinks-for-woocommerce' ),
				'id'       => self::SETTINGS_PREFIX . 'product_identifier',
				'default'  => 'default',
				'type'     => 'custom_permalinks_radio',
				'desc_tip' => __( 'You can choose what to use as a product slug, either a regular product slug (commonly generated from product title) or its SKU ', 'custom-permalinks-for-woocommerce' ),
				'options'  => array(
					'default' => array(
						'label' => __( 'Use defaults', 'custom-permalinks-for-woocommerce' ),
					),
					'sku'     => array(
						'label' => __( 'Product SKU', 'custom-permalinks-for-woocommerce' ),
					)
				),
			),

			array(
				'title'              => __( 'URL ending', 'custom-permalinks-for-woocommerce' ),
				'product_permalink'  => $this->get( 'product_permalink', 'default' ),
				'category_permalink' => $this->get( 'category_permalink', 'default' ),
				'product_identifier' => $this->getProductIdentifier(),
				'default'            => array(
					'ending'          => '',
					'enable_product'  => 'no',
					'enable_category' => 'no',
				),
				'type'               => 'custom_permalinks_url_ending',
				'desc_tip'           => __( 'You can append .html, .php, .htm or any other extension to the end of the URL to be displayed after the product or category slug ends.', 'custom-permalinks-for-woocommerce' ),
				'id'                 => self::SETTINGS_PREFIX . 'url_ending'
			),

			'main_section_end' => array(
				'type' => 'sectionend',
				'id'   => self::SETTINGS_PREFIX . 'main_section_end'
			),

			'advanced_section'     => array(
				'title' => __( 'Advanced', 'custom-permalinks-for-woocommerce' ),
				'desc'  => __( 'In case you\'re trying to handle permalinks on existing products and categories, you may wanna use rel-canonical and 301 redirects. When you enable 301 redirects that means if anyone goes to the old URL they\'ll be redirected to the corresponding new URL. And the rel=canonical will help you avoid duplicate pages from the search engine\'s point of view.',
					'custom-permalinks-for-woocommerce' ),
				'type'  => 'title',
			),
			array(
				'title'    => __( 'Product specific permalinks', 'custom-permalinks-for-woocommerce' ),
				'desc'     => __( 'Enable product specific settings per product', 'custom-permalinks-for-woocommerce' ),
				'id'       => self::SETTINGS_PREFIX . 'product_specific_permalinks',
				'default'  => 'no',
				'type'     => 'checkbox',
				'autoload' => false,
			),
			array(
				'title'    => __( '301 redirects', 'custom-permalinks-for-woocommerce' ),
				'desc'     => __( 'Apply 301 redirects to all URLs that have been changed', 'custom-permalinks-for-woocommerce' ),
				'id'       => self::SETTINGS_PREFIX . '301_redirects_enabled',
				'default'  => 'yes',
				'type'     => 'checkbox',
				'autoload' => false,
			),
			array(
				'title'    => __( 'Add canonicals', 'custom-permalinks-for-woocommerce' ),
				'desc'     => __( 'Apply ‘rel=canonical’ attribute to duplicate pages', 'custom-permalinks-for-woocommerce' ),
				'id'       => self::SETTINGS_PREFIX . 'canonicals',
				'default'  => 'yes',
				'type'     => 'checkbox',
				'autoload' => false,
			),
			array(
				'title'    => __( 'Use YOAST primary category', 'custom-permalinks-for-woocommerce' ),
				'desc'     => __( 'Apply primary category (as part of the permalink) from Yoast SEO', 'custom-permalinks-for-woocommerce' ),
				'id'       => self::SETTINGS_PREFIX . 'yoast_primary_category',
				'default'  => 'yes',
				'type'     => 'checkbox',
				'autoload' => false,
			),
			'advanced_section_end' => array(
				'type' => 'sectionend',
			)
		);


		$this->settings = apply_filters( 'custom_permalinks/settings/settings', $settings );
	}

	/**
	 * Add own settings tab
	 *
	 * @param array $settings_tabs
	 *
	 * @return mixed
	 */
	public function addSettingsTab( $settings_tabs ) {

		$settings_tabs[ self::SETTINGS_PAGE ] = __( 'Custom permalinks', 'custom-permalinks-for-woocommerce' );

		return $settings_tabs;
	}

	/**
	 * Add settings to WooCommerce
	 */
	public function registerSettings() {

		wp_enqueue_script( 'custom-permalinks-admin-input-mask-js', $this->getContainer()->getFileManager()->locateAsset( 'admin/settings.js' ), array( 'jquery' ), PermalinksPlugin::VERSION );
		wp_enqueue_script( 'custom-permalinks-admin-settings-js', $this->getContainer()->getFileManager()->locateAsset( 'admin/input-mask.min.js' ), array( 'jquery' ), PermalinksPlugin::VERSION );

		woocommerce_admin_fields( $this->settings );
	}

	/**
	 * Get setting by name
	 *
	 * @param string $option_name
	 * @param mixed $default
	 *
	 * @return mixed
	 */
	public function get( $option_name, $default = null ) {
		return get_option( self::SETTINGS_PREFIX . $option_name, $default );
	}


	public function getAllSettings() {

		$settings = array_filter( $this->settings, function ( $setting ) {
			return ! in_array( $setting['type'], array( 'section', 'sectionend', 'title' ) );
		} );

		return array_map( function ( $key, $value ) {
			return $this->get( $key, $value['default'] );
		}, array_keys( $settings ), $settings );

	}

	/**
	 * Get url to settings page
	 *
	 * @return string
	 */
	public function getLink() {
		return admin_url( 'admin.php?page=wc-settings&tab=' . self::SETTINGS_PAGE );
	}

	/**
	 * Sanitize URL ending option
	 *
	 * @param $value
	 *
	 * @return array
	 */
	public function sanitizeURLEnding( $value ) {
		$value['ending']          = isset( $value['ending'] ) ? ( trim( $value['ending'] ) ) : '';
		$value['enable_product']  = isset( $value['enable_product'] ) && 'yes' === $value['enable_product'] ? 'yes' : 'no';
		$value['enable_category'] = isset( $value['enable_category'] ) && 'yes' === $value['enable_category'] ? 'yes' : 'no';

		FlushService::setFlushRules();

		return $value;
	}

	public function isPermalinkEnabledForProducts( $specificProduct = null ) {
		return $this->getProductPermalink( $specificProduct ) !== 'default';
	}

	public function isPermalinkEnabledForCategory() {
		return $this->get( 'category_permalink', 'default' ) !== 'default';
	}

	public function getURLEnding() {
		$data = $this->get( 'url_ending', $this->sanitizeURLEnding( array() ) );

		return isset( $data['ending'] ) ? $data['ending'] : '';
	}

	public function isURLEndingEnabledForProducts() {
		$data = $this->get( 'url_ending', $this->sanitizeURLEnding( array() ) );

		return isset( $data['enable_product'] ) ? 'yes' === $data['enable_product'] : false;
	}

	public function isURLEndingEnabledForCategories() {
		$data = $this->get( 'url_ending', $this->sanitizeURLEnding( array() ) );

		return isset( $data['enable_category'] ) ? 'yes' === $data['enable_category'] : false;
	}

	public function isProductSpecificPermalinksEnabled() {
		return 'yes' === $this->get( 'product_specific_permalinks', 'no' );
	}

	public function getProductPermalink( $specificProduct = null ) {
		$globalSettings = $this->getContainer()->getSettings()->get( 'product_permalink', 'default' );

		if ( $specificProduct ) {
			$productSpecificPermalink = $this->getContainer()->getProductSpecificPermalinkService()->getProductSpecificPermalinkOption( $specificProduct );

			if ( ProductSpecificPermalinks::DEFAULT_VALUE !== $productSpecificPermalink ) {
				return $productSpecificPermalink;
			}
		}

		return $globalSettings;
	}
}
