<?php
/**
 * Custom Permalinks Taxonomies.
 *
 * @package CustomPermalinks
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Taxonomies Permalinks table class.
 */
final class Custom_Permalinks_Taxonomies {
	/**
	 * Sort the terms array in desc order using term id.
	 *
	 * @since 1.2.0
	 * @access public
	 *
	 * @param int $comp1 Value 1.
	 * @param int $comp2 Value 2.
	 *
	 * @return int
	 */
	public static function sort_array( $comp1, $comp2 ) {
		return $comp2['id'] - $comp1['id'];
	}

	/**
	 * Returns the count of records in the database.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return null|int
	 */
	public static function total_permalinks() {
		global $wpdb;

		$total_taxonomies = wp_cache_get( 'total_taxonomies_result', 'custom_permalinks' );
		if ( ! $total_taxonomies ) {
			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			// Include search in total results.
			if ( isset( $_REQUEST['s'] ) && ! empty( $_REQUEST['s'] ) ) {
				$search_value = ltrim( sanitize_text_field( $_REQUEST['s'] ), '/' );
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$total_taxonomies = $wpdb->get_var(
					$wpdb->prepare(
						"SELECT COUNT(t.term_id) FROM {$wpdb->terms} AS t
						LEFT JOIN {$wpdb->termmeta} AS tm ON (t.term_id = tm.term_id)
						WHERE tm.meta_key = 'custom_permalink'
							AND tm.meta_value != ''
							AND tm.meta_value LIKE %s",
						'%' . $wpdb->esc_like( $search_value ) . '%'
					)
				);
				// phpcs:enable WordPress.Security.NonceVerification.Recommended
			} else {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$total_taxonomies = $wpdb->get_var(
					"SELECT COUNT(t.term_id) FROM {$wpdb->terms} AS t
					LEFT JOIN {$wpdb->termmeta} AS tm ON (t.term_id = tm.term_id)
					WHERE tm.meta_key = 'custom_permalink' AND tm.meta_value != ''"
				);
			}

			wp_cache_set( 'total_taxonomies_result', $total_taxonomies, 'custom_permalinks' );
		}

		return $total_taxonomies;
	}

	/**
	 * Retrieve permalink's data from the database.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @param int $per_page    Maximum Results needs to be shown on the page.
	 * @param int $page_number Current page.
	 *
	 * @return array Title, Term Type and Permalink set using this plugin.
	 */
	public static function get_permalinks( $per_page = 20, $page_number = 1 ) {
		global $wpdb;

		$taxonomies = wp_cache_get( 'taxonomies_results', 'custom_permalinks' );
		if ( ! $taxonomies ) {
			$page_offset = ( $page_number - 1 ) * $per_page;
			$order_by    = 't.term_id';
			$order       = null;

			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			// Sort the items.
			switch ( isset( $_REQUEST['orderby'] ) ? strtolower( sanitize_text_field( $_REQUEST['orderby'] ) ) : '' ) {
				case 'title':
					$order_by = 't.name';
					break;

				case 'type':
					$order_by = 'ttm.taxonomy';
					break;

				case 'permalink':
					$order_by = 'tm.meta_value';
					break;
			}

			switch ( isset( $_REQUEST['order'] ) ? strtolower( sanitize_text_field( $_REQUEST['order'] ) ) : '' ) {
				case 'asc':
					$order = 'ASC';
					break;

				case 'desc':
				default:
					$order = 'DESC';
					break;
			}

			// Include search in total results.
			if ( isset( $_REQUEST['s'] ) && ! empty( $_REQUEST['s'] ) ) {
				$search_value = ltrim( sanitize_text_field( $_REQUEST['s'] ), '/' );
				// phpcs:disable WordPress.DB.PreparedSQLPlaceholders.UnquotedComplexPlaceholder
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$taxonomies = $wpdb->get_results(
					$wpdb->prepare(
						"SELECT t.term_id, t.name, tm.meta_value, ttm.taxonomy
							FROM {$wpdb->terms} AS t
						LEFT JOIN {$wpdb->termmeta} AS tm ON (t.term_id = tm.term_id)
						LEFT JOIN {$wpdb->term_taxonomy} AS ttm ON (t.term_id = ttm.term_id)
						WHERE tm.meta_key = 'custom_permalink'
							AND tm.meta_value != ''
							AND tm.meta_value LIKE %s
						ORDER BY %1s %2s LIMIT %d, %d",
						'%' . $wpdb->esc_like( $search_value ) . '%',
						$order_by,
						$order,
						$page_offset,
						$per_page
					),
					ARRAY_A
				);
				// phpcs:enable WordPress.DB.PreparedSQLPlaceholders.UnquotedComplexPlaceholder
			} else {
				// phpcs:disable WordPress.DB.PreparedSQLPlaceholders.UnquotedComplexPlaceholder
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$taxonomies = $wpdb->get_results(
					$wpdb->prepare(
						"SELECT t.term_id, t.name, tm.meta_value, ttm.taxonomy
							FROM {$wpdb->terms} AS t
						LEFT JOIN {$wpdb->termmeta} AS tm ON (t.term_id = tm.term_id)
						LEFT JOIN {$wpdb->term_taxonomy} AS ttm ON (t.term_id = ttm.term_id)
						WHERE tm.meta_key = 'custom_permalink' AND tm.meta_value != ''
						ORDER BY %1s %2s LIMIT %d, %d",
						$order_by,
						$order,
						$page_offset,
						$per_page
					),
					ARRAY_A
				);
				// phpcs:enable WordPress.DB.PreparedSQLPlaceholders.UnquotedComplexPlaceholder
			}
			// phpcs:enable WordPress.Security.NonceVerification.Recommended

			wp_cache_set( 'taxonomies_results', $taxonomies, 'custom_permalinks' );
		}

		return $taxonomies;
	}
}
