<?php
/**
 * Database related code.
 *
 * @package RedParts\Sputnik
 * @since 1.10.0
 */

namespace RedParts\Sputnik;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Database' ) ) {
	/**
	 * Class Database
	 *
	 * @since 1.10.0
	 */
	class Database extends Singleton {
		const VERSION_OPTION = 'redparts_sputnik_database_version';

		/**
		 * Migrations list.
		 *
		 * @since 1.10.0
		 *
		 * @var array
		 */
		public $migrations = array();

		/**
		 * Initialization.
		 *
		 * @since 1.10.0
		 */
		public function init() {
			add_action( 'init', array( $this, 'deferred_init' ), 5 );

			$this->migrations = array(
				'1.10.0' => array(
					array( $this, 'migrate_to_1_10_0_create_vehicle_fields_table' ),
				),
			);
		}

		/**
		 * Deferred initialization.
		 *
		 * @since 1.10.0
		 */
		public function deferred_init() {
			$version_option = get_option( self::VERSION_OPTION, '0.0.0' );

			if ( version_compare( $version_option, Plugin::VERSION, '>=' ) ) {
				return;
			}

			// Check if we are not already running this routine.
			if ( 'yes' === get_transient( 'redparts_sputnik_migrate_database' ) ) {
				return;
			}

			// If we made it till here nothing is running yet, lets set the transient now.
			set_transient( 'redparts_sputnik_migrate_database', 'yes', MINUTE_IN_SECONDS * 10 );

			foreach ( $this->migrations as $migration_version => $migration_functions ) {
				if ( version_compare( $version_option, $migration_version, '>=' ) ) {
					continue;
				}

				foreach ( $migration_functions as $migration_function ) {
					$migration_function();
				}

				delete_option( self::VERSION_OPTION );
				add_option( self::VERSION_OPTION, $migration_version );
			}

			Vehicles::instance()->reindex_vehicle_fields_table();
			Vehicles::instance()->clear_get_options_cache();

			delete_option( self::VERSION_OPTION );
			add_option( self::VERSION_OPTION, Plugin::VERSION );

			delete_transient( 'redparts_sputnik_migrate_database' );
		}

		/**
		 * Creates vehicle fields table.
		 *
		 * @since 1.10.0
		 */
		public function migrate_to_1_10_0_create_vehicle_fields_table() {
			global $wpdb;

			require_once ABSPATH . 'wp-admin/includes/upgrade.php';

			$collate = $wpdb->has_cap( 'collation' ) ? $wpdb->get_charset_collate() : '';

			$sql = "
			CREATE TABLE {$wpdb->prefix}redparts_sputnik_vehicle_fields (
				term_id BIGINT(20) UNSIGNED NOT NULL,
				field_key VARCHAR(255) NOT NULL,
				field_value VARCHAR(255) NULL DEFAULT NULL,
				PRIMARY KEY (term_id, field_key(64)),
				INDEX field_key_field_value (field_key(64), field_value(96))
			) $collate;
			";

			dbDelta( $sql );
		}
	}
}
